/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.deployment;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.FLOW_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestDomain;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static java.util.Collections.emptyMap;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.anypoint.tita.environment.api.artifact.DomainJar;
import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class PolicyWithPointcutsDeploymentTestCase extends AbstractMuleTestCase {

  private static final String EMPTY_API = "empty";
  private static final String SET_PAYLOAD_API = "set-payload";
  private static final String STAR_PATH_API = "star-path";

  private static DynamicPort appPort1 = new DynamicPort("port");
  private static DynamicPort domainPort = new DynamicPort("domainPort");

  private static SystemProperty policyPayload = new SystemProperty("policyPayload", POLICY_PAYLOAD);
  private static SystemProperty policyPayload2 = new SystemProperty("policyPayload2", POLICY_PAYLOAD_2);

  private static DomainJar domain = buildTestDomain("domain", "mule-domain-http.xml");
  private static ApplicationJar application1 = buildTestApplication(APP, "mule-config-http.xml");
  private static ApplicationJar application2 = buildTestApplication(APP_2, "mule-config-http-domain.xml", domain);

  private static FakeGatewayInstallation installation =
      builder()
          .withDomains(domain)
          .withApplications(application1,
                            application2)
          .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY, "policies/policy-deployment-policy.xml"),
                               buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_2, "policies/policy-deployment-policy-2.xml"))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(appPort1)
      .around(domainPort)
      .around(policyPayload)
      .around(policyPayload2)
      .around(installation);

  private final ApiFinder apiFinder;
  private final DynamicPort port;
  private final HttpRequest emptyRequest;
  private final HttpRequest setPayloadRequest;

  private FakeGatewayServer server = installation.getServer();

  public PolicyWithPointcutsDeploymentTestCase(String name, DynamicPort port, ApplicationJar application) {
    this.apiFinder = new ApiFinder(application.getAppConfig());
    this.port = port;
    this.emptyRequest = request(port, "/api/empty");
    this.setPayloadRequest = request(port, "/api/set-payload");
  }

  @Parameterized.Parameters(name = "Application {0}")
  public static Object[][] parameters() {
    return new Object[][] {
        {"without domain", appPort1, application1},
        {"with domain", domainPort, application2},
    };
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void apiPointcutAppliedOnlyToApiFlow() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, EMPTY_API);

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, emptyRequest.get().asString());
    assertEquals("Policy applied to incorrect request", FLOW_PAYLOAD, setPayloadRequest.get().asString());
  }

  @Test
  public void multiplePoliciesWithApiPointcuts() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, EMPTY_API);
    PolicyDefinition policyDefinition2 = policyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY_2, SET_PAYLOAD_API);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    assertEquals("First policy not applied to desired request", POLICY_PAYLOAD, emptyRequest.get().asString());
    assertEquals("Second policy not applied to desired request", FLOW_PAYLOAD + POLICY_PAYLOAD_2,
                 setPayloadRequest.get().asString());
  }

  @Test
  public void resourcePointcutByMethod() {
    PolicyDefinition policyDefinition =
        policyDefinition(POLICY_TEMPLATE_KEY, EMPTY_API, new HttpResourcePointcut(".*", "GET"));

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, emptyRequest.get().asString());
    assertTrue("Policy applied to incorrect request", emptyRequest.post().asString().isEmpty());
  }

  @Test
  public void resourcePointcutByExactPath() {
    HttpRequest starPath = request(port, "/api/star/path");
    HttpRequest starInnerPath = request(port, "/api/star/path/inner");
    HttpRequest starAnotherPath = request(port, "/api/star/anotherPath");
    PolicyDefinition policyDefinition =
        policyDefinition(POLICY_TEMPLATE_KEY, STAR_PATH_API, new HttpResourcePointcut("/api/star/path", ".*"));

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, starPath.get().asString());
    assertEquals("Policy applied to undesired request", "", starInnerPath.get().asString());
    assertEquals("Policy applied to undesired request", "", starAnotherPath.get().asString());
  }

  @Test
  public void resourcePointcutByStarPath() {
    HttpRequest starPath = request(port, "/api/star/path");
    HttpRequest starInnerPath = request(port, "/api/star/path/inner");
    HttpRequest starAnotherPath = request(port, "/api/star/path1");
    PolicyDefinition policyDefinition =
        policyDefinition(POLICY_TEMPLATE_KEY, STAR_PATH_API, new HttpResourcePointcut("/api/star/path(/.*)?", ".*"));

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, starPath.get().asString());
    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, starInnerPath.get().asString());
    assertEquals("Policy applied to undesired request", "", starAnotherPath.get().asString());
  }

  @Test
  public void resourcePointcutAppliedOnlyToApiFlow() {
    PolicyDefinition policyDefinition =
        policyDefinition(POLICY_TEMPLATE_KEY, EMPTY_API, new HttpResourcePointcut(".*", ".*"));

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, emptyRequest.get().asString());
    assertEquals("Policy applied to incorrect request", FLOW_PAYLOAD, setPayloadRequest.get().asString());
  }

  @Test
  public void multipleResourcePointcuts() {
    HttpRequest starPath = request(port, "/api/star/some/path");
    HttpRequest starAnotherPath = request(port, "/api/star/another/path");
    HttpRequest starNonMatching = request(port, "/api/star/non/matching");
    PolicyDefinition policyDefinition =
        policyDefinition(POLICY_TEMPLATE_KEY, STAR_PATH_API, new HttpResourcePointcut("/api/star/some/path", ".*"),
                         new HttpResourcePointcut("/api/star/another/path", ".*"));

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, starPath.get().asString());
    assertEquals("Policy not applied to desired request", POLICY_PAYLOAD, starAnotherPath.get().asString());
    assertEquals("Policy applied to undesired request", "", starNonMatching.get().asString());
  }

  private PolicyDefinition policyDefinition(PolicyTemplateKey key, String apiFlow, HttpResourcePointcut resourcePointcut,
                                            HttpResourcePointcut resourcePointcut2) {
    return new PolicyDefinition(POLICY_ID, key, apiFinder.find(apiFlow), newArrayList(resourcePointcut, resourcePointcut2), 1,
                                new PolicyConfiguration(emptyMap()));
  }

  private PolicyDefinition policyDefinition(PolicyTemplateKey templateKey, String apiFlow) {
    return new PolicyDefinition(POLICY_ID, templateKey, apiFinder.find(apiFlow), null, 1, new PolicyConfiguration(emptyMap()));
  }

  private PolicyDefinition policyDefinition(PolicyTemplateKey templateKey, String apiFlow,
                                            HttpResourcePointcut resourcePointcut) {
    return new PolicyDefinition(POLICY_ID, templateKey, apiFinder.find(apiFlow), newArrayList(resourcePointcut), 1,
                                new PolicyConfiguration(emptyMap()));
  }

  private PolicyDefinition policyDefinition(String id, PolicyTemplateKey templateKey, String apiFlow) {
    return new PolicyDefinition(id, templateKey, apiFinder.find(apiFlow), null, 1, new PolicyConfiguration(emptyMap()));
  }

}
