/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.deployment;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_3;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestDomain;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static java.util.Collections.emptyMap;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.isEmptyString;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.anypoint.tita.environment.api.artifact.DomainJar;
import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.Arrays;
import java.util.Collection;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class ResourceLevelPolicyIgnoreBasePathDeploymentTestCase extends AbstractMuleTestCase {

  private static final String ENABLED_API = "enabled";
  private static final String DISABLED_API = "disabled";
  private static final String NO_WILDCARD_API = "no-wildcard";
  private static final String WILDCARD_IN_MIDDLE_API = "wildcard-in-middle";

  private static DynamicPort portApp1 = new DynamicPort("port");
  private static DynamicPort portApp2 = new DynamicPort("port2");
  private static DynamicPort domainPort = new DynamicPort("domainPort");

  private static SystemProperty policyPayload = new SystemProperty("policyPayload", POLICY_PAYLOAD);

  private static DomainJar domain = buildTestDomain("domain", "mule-domain-http-2.xml");
  private static ApplicationJar application1 = buildTestApplication(APP, "mule-config-ignore-base-path-annotation.xml");
  private static ApplicationJar application2 = buildTestApplication(APP_2, "mule-config-ignore-base-path-attribute.xml");
  private static ApplicationJar application3 =
      buildTestApplication(APP_3, "mule-config-ignore-base-path-attribute-domain.xml", domain);

  private static FakeGatewayInstallation installation =
      builder()
          .withDomains(domain)
          .withApplications(application1,
                            application2,
                            application3)
          .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY, "policies/policy-deployment-policy.xml"))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(portApp1)
      .around(portApp2)
      .around(domainPort)
      .around(policyPayload)
      .around(installation);

  private HttpRequest enabledRequest;
  private HttpRequest disabledRequest;
  private HttpRequest noWildcardRequest;
  private HttpRequest wildcardInMiddleRequest;
  private ApiFinder apiFinder;
  private DynamicPort port;

  private FakeGatewayServer server = installation.getServer();

  @Parameterized.Parameters(name = "Application {0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {"with ignoreBasePath annotation", portApp1, application1},
        {"with ignoreBasePath attribute", portApp2, application2},
        {"with ignoreBasePath attribute and domain", domainPort, application3}
    });
  }

  public ResourceLevelPolicyIgnoreBasePathDeploymentTestCase(String name, DynamicPort dynamicPort, ApplicationJar application) {
    this.enabledRequest = request(dynamicPort, "/base/path/enabled/api/resource");
    this.disabledRequest = request(dynamicPort, "/base/path/disabled/api/resource");
    this.noWildcardRequest = request(dynamicPort, "/base/path/no-wildcard/api/resource");
    this.wildcardInMiddleRequest = request(dynamicPort, "/base/path/wildcard-in-middle/api/resource");
    this.apiFinder = new ApiFinder(application.getAppConfig());
    this.port = dynamicPort;
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void ignoreBasePathWorks() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut("/api/.*", ".*");
    PolicyDefinition policyDefinition = policyDefinition(ENABLED_API, pointcut);

    server.deployPolicy(policyDefinition);

    assertThat(enabledRequest.get().asString(), is(POLICY_PAYLOAD));
  }

  @Test
  public void oldDoesNotWork() {
    HttpResourcePointcut oldPointcut = new HttpResourcePointcut("/base/path/enabled/.*", ".*");
    PolicyDefinition policyDefinition = policyDefinition(ENABLED_API, oldPointcut);

    server.deployPolicy(policyDefinition);

    assertThat(enabledRequest.get().asString(), isEmptyString());
    assertThat(request(port, "/base/path/enabled/base/path/enabled/").get().asString(), is(POLICY_PAYLOAD));
  }

  @Test
  public void annotationPresentWithFalseValueWorksAsBefore() {
    HttpResourcePointcut oldPointcut = new HttpResourcePointcut("/base/path/disabled/.*", ".*");
    PolicyDefinition policyDefinition = policyDefinition(DISABLED_API, oldPointcut);

    server.deployPolicy(policyDefinition);

    assertThat(disabledRequest.get().asString(), is(POLICY_PAYLOAD));
  }

  @Test
  public void noWildcardInListenerWorksAsBefore() {
    HttpResourcePointcut oldPointcut = new HttpResourcePointcut("/base/path/no-wildcard/.*", ".*");
    PolicyDefinition policyDefinition = policyDefinition(NO_WILDCARD_API, oldPointcut);

    server.deployPolicy(policyDefinition);

    assertThat(noWildcardRequest.get().asString(), is(POLICY_PAYLOAD));
  }

  @Test
  public void wildcardInMiddleWorksAsBefore() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut("/base/path/wildcard-in-middle/api/.*", ".*");
    PolicyDefinition policyDefinition = policyDefinition(WILDCARD_IN_MIDDLE_API, pointcut);

    server.deployPolicy(policyDefinition);

    assertThat(wildcardInMiddleRequest.get().asString(), is(POLICY_PAYLOAD));
  }

  private PolicyDefinition policyDefinition(String apiFlow, HttpResourcePointcut resourcePointcut) {
    return new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, apiFinder.find(apiFlow), newArrayList(resourcePointcut), 1,
                                new PolicyConfiguration(emptyMap()));
  }

}
