/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.error;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.GROUP_ID;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.TestDependencies.testAuthenticationDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static java.util.Collections.emptyList;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.Map;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

/**
 * Tests for asserting that security context is always propagated even when errors are thrown from policies or flow/operations.
 *
 * In every test a witness policy is deployed along with the actual policy that throws an error. In this case both policies are
 * adding a property in the authentication object in each part of the policy that is executed. At the end, those properties are
 * returned to the calling client as headers by the witness policy.
 * 
 * In this way it can be asserted that all the properties were propagated throughout the complete chain.
 */
@RunWith(Parameterized.class)
public class PolicyErrorHandlingSecurityContextTestCase extends AbstractMuleTestCase {

  private static final PolicyTemplateKey OPERATION_RAISE_ERROR =
      new PolicyTemplateKey(GROUP_ID, "OperationExceptionAndSecurityContext", "0.1.0");
  private static final PolicyTemplateKey ASSERTER_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "AssertSecurityContext", "0.1.0");

  private static DynamicPort portApp1 = new DynamicPort("port");
  private static DynamicPort portApp2 = new DynamicPort("port2");
  private static DynamicPort implementationPort = new DynamicPort("implementation.port");

  private static ApplicationJar app1 =
      buildTestApplication(APP, "mule-config-error-handling-operation-security-context.xml", testAuthenticationDependency());
  private static ApplicationJar app2 =
      buildTestApplication(APP_2, "mule-config-error-handling-operation-security-context-target.xml",
                           testAuthenticationDependency());

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(app1,
                            app2,
                            buildTestApplication("server", "mule-config-error-handling-operation-external-server.xml"))
          .withPolicyTemplates(
                               buildTestPolicyTemplate(OPERATION_RAISE_ERROR,
                                                       "templates/operation/exception-in-operation-and-sending-security-context-template.xml",
                                                       testAuthenticationDependency()),
                               buildTestPolicyTemplate(ASSERTER_TEMPLATE_ID,
                                                       "policies/policy-operation-and-source-scope-security-context-policy.xml",
                                                       testAuthenticationDependency()))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain chain = RuleChain.outerRule(portApp1)
      .around(portApp2)
      .around(new SystemProperty("serverPayload", "Server Payload"))
      .around(implementationPort)
      .around(installation);

  private final ApiFinder apiFinder;

  private HttpRequest setPayloadRequest;
  private HttpRequest errorInFlowRequest;
  private HttpRequest errorInOperationRequest;

  protected PolicyDefinition policyDefinition;
  protected PolicyDefinition policyDefinition2;

  protected FakeGatewayServer server = installation.getServer();

  private Boolean propagateMessageTransformations;
  private SerialExecutor serialExecutor = new SerialExecutor(Runtime.getRuntime().availableProcessors() + 1);

  public PolicyErrorHandlingSecurityContextTestCase(Boolean propagateMessageTransformations, DynamicPort port,
                                                    ApplicationJar application) {
    this.propagateMessageTransformations = propagateMessageTransformations;
    this.setPayloadRequest = request(port, "/set-payload/server-resource");
    this.errorInFlowRequest = request(port, "/error-in-flow/server-resource");
    this.errorInOperationRequest = request(port, "/error-in-operation/server-resource");
    this.apiFinder = new ApiFinder(application.getAppConfig());
  }

  @Parameterized.Parameters(name = "Propagating message modification: {0}")
  public static Object[][] parameters() {
    return new Object[][] {
        {false, portApp1, app1},
        {true, portApp1, app1},
        {false, portApp2, app2},
        {true, portApp2, app2}
    };
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  /**
   * Error thrown BEFORE the operation is executed in a policy which defines an on-error-continue handler
   */
  @Test
  public void raiseBeforeOpErrorContinue() {

    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                         config().errorContinue()
                                             .errorBefore()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(200));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("afterOp-p2"), nullValue());
      assertThat(response.header("afterOp-p1"), is("true"));
      assertThat(response.header("opEh-p2"), is("true"));
      assertThat(response.header("opEh-p1"), nullValue());
      assertThat(response.header("flow"), is("true"));
      assertThat(response.header("flow-eh"), nullValue());
      assertThat(response.header("afterSource-p2"), is("true"));
      assertThat(response.header("afterSource-p1"), is("true"));
      assertThat(response.header("sourceEh-p2"), nullValue());
      assertThat(response.header("sourceEh-p1"), nullValue());
    });
  }

  /**
   * Error thrown AFTER the operation is executed in a policy which defines an on-error-continue handler
   */
  @Test
  public void raiseAfterOpErrorContinue() {
    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                         config().errorContinue()
                                             .errorAfter()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);


    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(200));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("afterOp-p1"), is("true"));
      assertThat(response.header("afterOp-p2"), is("true"));
      assertThat(response.header("opEh-p2"), is("true"));
      assertThat(response.header("afterSource-p1"), is("true"));
      assertThat(response.header("afterSource-p2"), is("true"));
      assertThat(response.header("flow"), is("true"));
    });
  }

  /**
   * Error thrown BEFORE the operation is executed in a policy which defines an on-error-propagate handler
   */
  @Test
  public void raiseBeforeErrorPropagate() {
    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                         config().errorPropagate()
                                             .errorBefore()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("opEh-p1"), is("true"));
      assertThat(response.header("opEh-p2"), is("true"));
      assertThat(response.header("sourceEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p2"), is("true"));
      assertThat(response.header("flow"), is("true"));
      assertThat(response.header("flow-eh"), is("true"));
    });
  }

  /**
   * Error thrown AFTER the operation is executed in a policy which defines an on-error-propagate handler
   */
  @Test
  public void raiseAfterErrorPropagate() {
    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                         config().errorPropagate()
                                             .errorAfter()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("afterOp-p2"), is("true"));
      assertThat(response.header("opEh-p1"), is("true"));
      assertThat(response.header("opEh-p2"), is("true"));
      assertThat(response.header("sourceEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p2"), is("true"));
      assertThat(response.header("flow"), is("true"));
      assertThat(response.header("flow-eh"), is("true"));
    });
  }

  /**
   * Error thrown BEFORE the operation is executed in a policy which does not define an error handler
   */
  @Test
  public void raiseBeforeNoErrorHandler() {
    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                         config().errorContinue()
                                             .noErrorHandler()
                                             .errorBefore()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("opEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p2"), is("true"));
      assertThat(response.header("flow"), is("true"));
      assertThat(response.header("flow-eh"), is("true"));
    });
  }

  /**
   * Error thrown AFTER the operation is executed in a policy which does not define an error handler
   */
  @Test
  public void raiseAfterNoErrorHandler() {
    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                         config().noErrorHandler()
                                             .errorAfter()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("afterOp-p2"), is("true"));
      assertThat(response.header("opEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p2"), is("true"));
      assertThat(response.header("flow"), is("true"));
      assertThat(response.header("flow-eh"), is("true"));
    });
  }

  /**
   * Error thrown in the flow AFTER the operation is executed. Policy's error handler does not matter since it won't be catched by
   * it
   */
  @Test
  public void errorInFlow() {
    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInFlowApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, errorInFlowApiKey(),
                                         config().errorContinue()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInFlowRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("afterOp-p1"), is("true"));
      assertThat(response.header("afterOp-p2"), is("true"));
      assertThat(response.header("sourceEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p2"), is("true"));
      assertThat(response.header("flow"), is("true"));
      assertThat(response.header("flow-eh"), is("true"));
    });
  }

  /**
   * Error thrown while the operation is executed.
   */
  @Test
  public void errorInOperation() {
    policyDefinition = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInOperationApiKey(), asserterConfig());

    policyDefinition2 = policyDefinition(2, OPERATION_RAISE_ERROR, errorInOperationApiKey(),
                                         config().errorPropagate()
                                             .get());

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInOperationRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.header("beforeSource-p1"), is("true"));
      assertThat(response.header("beforeSource-p2"), is("true"));
      assertThat(response.header("beforeOp-p1"), is("true"));
      assertThat(response.header("beforeOp-p2"), is("true"));
      assertThat(response.header("opEh-p1"), is("true"));
      assertThat(response.header("opEh-p2"), is("true"));
      assertThat(response.header("sourceEh-p1"), is("true"));
      assertThat(response.header("sourceEh-p2"), is("true"));
      assertThat(response.header("flow"), is("true"));
      assertThat(response.header("flow-eh"), is("true"));
    });
  }

  public PolicyDefinition policyDefinition(int order, PolicyTemplateKey templateKey, ApiKey apiKey,
                                           Map<String, Object> configData) {
    return new PolicyDefinition(String.valueOf(order), templateKey, apiKey, emptyList(), order,
                                new PolicyConfiguration(configData));
  }

  public Config config() {
    return new Config("p2", propagateMessageTransformations);
  }

  private Map<String, Object> asserterConfig() {
    return of("policyId", "p1",
              "propagationEnabled", propagateMessageTransformations);
  }

  private ApiKey setPayloadApiKey() {
    return apiFinder.find("set-payload");
  }

  private ApiKey errorInFlowApiKey() {
    return apiFinder.find("error-in-flow");
  }

  private ApiKey errorInOperationApiKey() {
    return apiFinder.find("error-in-operation");
  }
}
