/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.error;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_3;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.GROUP_ID;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.TestDependencies.testAssertionsDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.Map;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

/**
 * Tests for asserting that policy variables are isolated from other policies and the flow/operation when errors are thrown from
 * policies and from the flow.
 * 
 * In every test a witness policy is deployed along with the actual policy that throws an error to verify that variables defined
 * in the other one are not visible
 */
@RunWith(Parameterized.class)
public class PolicyErrorHandlingVariablesTestCase extends AbstractMuleTestCase {

  private static final PolicyTemplateKey SOURCE_RAISE_ERROR =
      new PolicyTemplateKey(GROUP_ID, "SourceExceptionAndVariable", "0.1.0");
  private static final PolicyTemplateKey OPERATION_RAISE_ERROR =
      new PolicyTemplateKey(GROUP_ID, "OperationExceptionAndVariable", "0.1.0");
  private static final PolicyTemplateKey ASSERTER_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "ReadVar", "0.1.0");

  private static final String SUCCESS_PAYLOAD = "assertions success in asserter";

  private static DynamicPort portApp1 = new DynamicPort("port");
  private static DynamicPort portApp2 = new DynamicPort("port2");
  private static DynamicPort implementationPort = new DynamicPort("implementation.port");

  private static ApplicationJar app1 = buildTestApplication(APP, "mule-config-error-handling-operation-vars.xml",
                                                            testAssertionsDependency());
  private static ApplicationJar app2 = buildTestApplication(APP_2, "mule-config-error-handling-operation-vars-target.xml",
                                                            testAssertionsDependency());

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(app1,
                            app2,
                            buildTestApplication(APP_3, "mule-config-error-handling-operation-external-server.xml"))
          .withPolicyTemplates(
                               buildTestPolicyTemplate(SOURCE_RAISE_ERROR,
                                                       "templates/operation/exception-in-source-and-sending-variable-template.xml",
                                                       testAssertionsDependency()),
                               buildTestPolicyTemplate(OPERATION_RAISE_ERROR,
                                                       "templates/operation/exception-in-operation-and-sending-variable-template.xml",
                                                       testAssertionsDependency()),
                               buildTestPolicyTemplate(ASSERTER_TEMPLATE_ID,
                                                       "templates/operation/variable-asserter-template.xml",
                                                       testAssertionsDependency()))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain chain = RuleChain.outerRule(portApp1)
      .around(portApp2)
      .around(new SystemProperty("serverPayload", "Server Payload"))
      .around(implementationPort)
      .around(installation);

  private final ApiFinder apiFinder;

  private HttpRequest setPayloadRequest;
  private HttpRequest errorInFlowRequest;
  private HttpRequest errorInServer;

  private PolicyDefinition witness;
  private PolicyDefinition policyDefinition;

  private Boolean propagateMessageTransformations;

  private FakeGatewayServer server = installation.getServer();
  private SerialExecutor serialExecutor = new SerialExecutor(Runtime.getRuntime().availableProcessors() + 1);

  public PolicyErrorHandlingVariablesTestCase(Boolean propagateMessageTransformations, DynamicPort port,
                                              ApplicationJar application) {
    this.propagateMessageTransformations = propagateMessageTransformations;
    this.setPayloadRequest = request(port, "/set-payload/server-resource");
    this.errorInFlowRequest = request(port, "/error-in-flow/server-resource");
    this.errorInServer = request(port, "/error-in-server/server-resource");
    this.apiFinder = new ApiFinder(application.getAppConfig());
  }

  @Parameterized.Parameters(name = "Propagating message modification: {0}")
  public static Object[][] parameters() {
    return new Object[][] {
        {false, portApp1, app1},
        {true, portApp1, app1},
        {false, portApp2, app2},
        {true, portApp2, app2}
    };
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  /**
   * Error thrown BEFORE the operation is executed in a policy which defines an on-error-continue handler
   */
  @Test
  public void raiseBeforeOpErrorContinue() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                        config().errorContinue()
                                            .errorBefore()
                                            .shouldExecuteErrorHandler()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown AFTER the operation is executed in a policy which defines an on-error-continue handler
   */
  @Test
  public void raiseAfterOpErrorContinue() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                        config().errorContinue()
                                            .errorAfter()
                                            .shouldExecuteErrorHandler()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown BEFORE the operation is executed in a policy which defines an on-error-propagate handler
   */
  @Test
  public void raiseBeforeErrorPropagate() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInServerApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, errorInServerApiKey(),
                                        config().errorPropagate()
                                            .errorBefore()
                                            .shouldExecuteErrorHandler()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInServer.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown AFTER the operation is executed in a policy which defines an on-error-propagate handler
   */
  @Test
  public void raiseAfterErrorPropagate() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                        config().errorPropagate()
                                            .errorAfter()
                                            .shouldExecuteErrorHandler()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown BEFORE the operation is executed in a policy which does not define an error handler
   */
  @Test
  public void raiseBeforeNoErrorHandler() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInServerApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, errorInServerApiKey(),
                                        config().noErrorHandler()
                                            .errorBefore()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInServer.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown AFTER the operation is executed in a policy which does not define an error handler
   */
  @Test
  public void raiseAfterNoErrorHandler() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, setPayloadApiKey(),
                                        config().noErrorHandler()
                                            .errorAfter()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown in the flow AFTER the operation is executed. Policy's error handler does not matter since it won't be catched by
   * it
   */
  @Test
  public void errorInFlow() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInFlowApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, errorInFlowApiKey(),
                                        config().noErrorHandler().get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInFlowRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
      assertThat(response.header("flow-variable"), is("value"));
    });
  }

  /**
   * Error thrown by the actual operation and there is an operation policy with a on-error-continue handler defined
   */
  @Test
  public void errorInOperationErrorContinue() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInServerApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, errorInServerApiKey(),
                                        config().errorContinue()
                                            .shouldExecuteErrorHandler()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInServer.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown by the actual operation and there is an operation policy with a on-error-propagate handler defined
   */
  @Test
  public void errorInOperationErrorPropagate() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInServerApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, errorInServerApiKey(),
                                        config().errorPropagate()
                                            .shouldExecuteErrorHandler()
                                            .get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInServer.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown by the actual operation and there is an operation policy without an error handler defined
   */
  @Test
  public void errorInOperationNoErrorHandler() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, errorInServerApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, OPERATION_RAISE_ERROR, errorInServerApiKey(), config().noErrorHandler().get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInServer.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown in the source policy which defines an on-error-continue handler
   */
  @Test
  public void raiseErrorInSourceErrorContinue() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, SOURCE_RAISE_ERROR, setPayloadApiKey(),
                                        config().errorContinue().get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown in the source policy which defines an on-error-propagate handler
   */
  @Test
  public void raiseErrorInSourceErrorPropagate() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, SOURCE_RAISE_ERROR, setPayloadApiKey(),
                                        config().errorPropagate().get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  /**
   * Error thrown in the source policy which does not define an error handler
   */
  @Test
  public void raiseErrorInSourceNoErrorHandler() {
    witness = policyDefinition(1, ASSERTER_TEMPLATE_ID, setPayloadApiKey(), asserterConfig());

    policyDefinition = policyDefinition(2, SOURCE_RAISE_ERROR, setPayloadApiKey(),
                                        config().noErrorHandler().get());

    server.deployPolicy(witness);
    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(SUCCESS_PAYLOAD));
    });
  }

  public PolicyDefinition policyDefinition(int order, PolicyTemplateKey templateKey, ApiKey apiKey,
                                           Map<String, Object> configData) {
    return new PolicyDefinition(String.valueOf(order), templateKey, apiKey, null, order, new PolicyConfiguration(configData));
  }

  public Config config() {
    return new Config("p1", propagateMessageTransformations);
  }

  /**
   * otherId represents the id of the policy that is actually setting variables and throwing errors
   */
  private Map<String, Object> asserterConfig() {
    return of("otherId", "p1",
              "propagationEnabled", propagateMessageTransformations);
  }

  private ApiKey setPayloadApiKey() {
    return apiFinder.find("set-payload");
  }

  private ApiKey errorInFlowApiKey() {
    return apiFinder.find("error-in-flow");
  }

  private ApiKey errorInServerApiKey() {
    return apiFinder.find("error-in-server");
  }

}
