/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.error.source;

import static com.mulesoft.anypoint.test.policy.error.PolicyErrorHandlingScenarios.HeadersMode.DONT_INCLUDE_HEADERS;
import static com.mulesoft.anypoint.test.policy.error.PolicyErrorHandlingScenarios.HeadersMode.INCLUDE_HEADERS;
import static com.mulesoft.anypoint.test.policy.error.PolicyErrorHandlingScenarios.StatusCodeMode.DEFAULT_STATUS_CODE;
import static com.mulesoft.anypoint.test.policy.error.PolicyErrorHandlingScenarios.StatusCodeMode.OVERRIDE_STATUS_CODE;
import static com.mulesoft.anypoint.test.policy.error.source.PolicySourceErrorContinueTestCase.ConfigurationImpl.DEFAULT_SC_DONT_INCLUDE_HEADERS;
import static com.mulesoft.anypoint.test.policy.error.source.PolicySourceErrorContinueTestCase.ConfigurationImpl.DEFAULT_SC_INCLUDE_HEADERS;
import static com.mulesoft.anypoint.test.policy.error.source.PolicySourceErrorContinueTestCase.ConfigurationImpl.OVERRIDE_SC_DONT_INCLUDE_HEADERS;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import com.mulesoft.anypoint.tests.http.HttpResponse;

@RunWith(Parameterized.class)
public class PolicySourceErrorContinueTestCase extends PolicySourceErrorHandlingScenarios {

  @Parameterized.Parameters
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {DEFAULT_SC_DONT_INCLUDE_HEADERS},
        {OVERRIDE_SC_DONT_INCLUDE_HEADERS},
        {DEFAULT_SC_INCLUDE_HEADERS}
    });
  }

  public PolicySourceErrorContinueTestCase(Configuration config) {
    super(config);
  }

  /**
   * Asserts the generated status code when the error was thrown from a policy.
   *
   * If the policy's error-handler is updating the status code, then the HTTP response should contain the updated value.
   *
   * If the policy's error handler is not updating the status code, but is updating headers, then status code ends up being null,
   * and since the error is handled correctly, the default status code for successful chain execution is used, which is 200.
   *
   * If the policy's error handler is not updating the message, then it will contain the value that the next chain generated (flow
   * or another policy). If the next chain does not complete with any status code, then a 200 ends up being set because the policy
   * chain is completing without error due to the on-error-continue.
   */
  protected void assertVariableErrorStatusCode(HttpResponse response, int nextStatusCode) {
    int statusCode;

    if (OVERRIDE_STATUS_CODE.equals(config.getStatusCodeMode())) {
      statusCode = OVERRIDDEN_STATUS_CODE;
    } else if (INCLUDE_HEADERS.equals(config.getHeadersMode())) {
      statusCode = 200;
    } else {
      statusCode = nextStatusCode == NONE ? 200 : nextStatusCode;
    }

    assertThat(response.statusCode(), is(statusCode));
  }

  @Override
  protected String policyErrorHandlerPayload() {
    return POLICY_ERROR_PAYLOAD;
  }

  public static class ConfigurationImpl extends Configuration {

    /**
     * Error Handler defined in the policy does not updates the message at all
     */
    static final Configuration DEFAULT_SC_DONT_INCLUDE_HEADERS =
        new ConfigurationImpl(DEFAULT_STATUS_CODE, DONT_INCLUDE_HEADERS);

    /**
     * Error Handler defined in the policy updates the status code
     */
    static final Configuration OVERRIDE_SC_DONT_INCLUDE_HEADERS =
        new ConfigurationImpl(OVERRIDE_STATUS_CODE, DONT_INCLUDE_HEADERS);

    /**
     * Error Handler defined in the policy updates the status code and adds a new header
     */
    static final Configuration DEFAULT_SC_INCLUDE_HEADERS = new ConfigurationImpl(DEFAULT_STATUS_CODE, INCLUDE_HEADERS);

    ConfigurationImpl(StatusCodeMode statusCodeMode, HeadersMode headersMode) {
      super(statusCodeMode, headersMode);
    }

    public Map<String, Object> getConfigData() {
      Map<String, Object> result = new HashMap<>();
      result.put("errorContinue", true);
      result.put("errorPropagate", false);
      result.put("overrideStatusCode", OVERRIDE_STATUS_CODE.equals(getStatusCodeMode()));
      result.put("includeHeaders", INCLUDE_HEADERS.equals(getHeadersMode()));
      result.put("overrideStatusCodeOrIncludeHeaders",
                 OVERRIDE_STATUS_CODE.equals(getStatusCodeMode()) || INCLUDE_HEADERS.equals(getHeadersMode()));
      result.put("errorPayload", POLICY_ERROR_PAYLOAD);
      return result;
    }

  }
}
