/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.error.source;

import static com.mulesoft.anypoint.test.policy.error.PolicyErrorHandlingScenarios.HeadersMode.INCLUDE_HEADERS;
import static com.mulesoft.anypoint.test.policy.error.PolicyErrorHandlingScenarios.StatusCodeMode.DEFAULT_STATUS_CODE;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assume.assumeTrue;

import java.util.Map;

import org.junit.Test;

import com.mulesoft.anypoint.tests.http.HttpResponse;

import io.qameta.allure.Description;

public abstract class PolicySourceErrorHandlingScenarios extends PolicySourceErrorHandlingInfrastructure {

  /**
   * Status code is not present. This is because Mule's message attributes does not contain an HttpResponseAttributes, therefore
   * there is no status code in the message.
   */
  protected static final int NONE = 0;

  protected Configuration config;

  public PolicySourceErrorHandlingScenarios(Configuration config) {
    this.config = config;
  }

  @Description("One policy: error thrown before executing the flow")
  @Test
  public void onePolicyErrorBeforeFlow() {
    policyDefinition = policyDefinition(1, BEFORE_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(BEFORE_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, NONE);
    });
  }

  @Description("One policy: try within try, error thrown from the inner try which has an on-error-continue")
  @Test
  public void onePolicyErrorInNestedTryHandled() {
    Map<String, Object> configData = config.getConfigData();
    configData.put("secondErrorContinue", true);

    policyDefinition =
        policyDefinition(1, BEFORE_NEXT_NESTED_TRY_TEMPLATE_ID, SET_PAYLOAD_API_KEY, configData);

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      // Outside error-handler is never executed
      assertThat(response.asString(), is(BEFORE_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + AFTER_PAYLOAD));
      assertThat(response.statusCode(), is(200));
      assertThat(response.header("testHeader"), nullValue());
    });
  }

  @Description("One policy: try within try, error thrown from the inner try which has an on-error-propagate")
  @Test
  public void onePolicyErrorInNestedTryPropagated() {

    Map<String, Object> configData = config.getConfigData();
    configData.put("secondErrorPropagate", true);

    policyDefinition =
        policyDefinition(1, BEFORE_NEXT_NESTED_TRY_TEMPLATE_ID, SET_PAYLOAD_API_KEY, configData);

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.asString(), is(BEFORE_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, NONE);
    });
  }

  @Description("One policy: error thrown after executing the flow, overriding status code in the listener")
  @Test
  public void onePolicyExceptionAfterExecuteNextAndOverridingSC() {
    policyDefinition = policyDefinition(1, AFTER_NEXT_TEMPLATE_ID, SET_PAYLOAD_OVERRIDING_SC_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadOverridingStatusCodeRequest.get();

      assertVariableErrorStatusCode(response, 250);
      assertThat(response.asString(), is(FLOW_PAYLOAD + FLOW_ERROR_PAYLOAD + AFTER_PAYLOAD + policyErrorHandlerPayload()));
    });
  }

  @Description("One policy: error thrown after executing the flow")
  @Test
  public void onePolicyExceptionAfterExecuteNext() {
    policyDefinition = policyDefinition(1, AFTER_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.asString(), is(FLOW_PAYLOAD + AFTER_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, 200);
    });
  }

  @Description("One policy: error thrown inside the policy error handler, with a previous error before the flow thrown to reach that error "
      + "handler")
  @Test
  public void onePolicyExceptionInPolicyErrorHandler() {
    assumeTrue(DEFAULT_STATUS_CODE.equals(config.getStatusCodeMode()));

    policyDefinition = policyDefinition(1, IN_ERROR_HANDLER_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.statusCode(), is(500)); // Always 500 as it cannot be saved
      assertThat(response.asString(), containsString(BEFORE_PAYLOAD + policyErrorHandlerPayload()));
    });
  }

  @Description("One policy: error thrown inside the flow error handler, with a previous error thrown in the flow to reach that "
      + "error handler")
  @Test
  public void onePolicyExceptionInFlowErrorHandler() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, ERROR_IN_ERROR_HANDLER_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInErrorHandlerRequest.get();
      assertThat(response.asString(), is(FLOW_PAYLOAD + FLOW_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, 500);
    });
  }

  @Description("One policy: error thrown in the flow which does not have an error handler")
  @Test
  public void onePolicyExceptionInFlow() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, ERROR_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorRequest.get();
      assertVariableErrorStatusCode(response, 500);
      assertThat(response.asString(), is("Error in flow" + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
    });
  }

  @Description("One policy: error thrown in the response builder of the flow's listener")
  @Test
  public void onePolicyExceptionInResponseBuilder() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, ERROR_IN_RESPONSE_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInResponseBuilderRequest.get();
      assertHeaderIncluded(response);
      // Normal response builder explodes and the error is catched by the error response builder
      assertThat(response.asString(), is(FLOW_PAYLOAD + FLOW_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertVariableErrorStatusCode(response, 450);
    });
  }

  @Description("One policy: error thrown after executing the flow, overriding status code in the listener")
  @Test
  public void onePolicyExceptionInFlowWithErrorResponseBuilder() {
    policyDefinition =
        policyDefinition(1, AFTER_NEXT_TEMPLATE_ID, ERROR_RESPONSE_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorResponseRequest.get();

      assertVariableErrorStatusCode(response, 450);
      assertThat(response.asString(), is(FLOW_PAYLOAD + FLOW_ERROR_PAYLOAD + policyErrorHandlerPayload()));
    });
  }

  @Description("One policy: error thrown in the error response builder of the flow's listener, with a previous error thrown to "
      + "reach that builder")
  @Test
  public void onePolicyExceptionInErrorResponseBuilder() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, ERROR_IN_ERROR_RESPONSE_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorInErrorResponseBuilderRequest.get();
      // Flow finishes without status code because response builder explodes when trying to be built
      assertVariableErrorStatusCode(response, NONE);
      assertThat(response.asString(), is(FLOW_PAYLOAD + policyErrorHandlerPayload()));
    });
  }

  @Description("One policy: error thrown in the flow which has defined an error handler with on-error-continue")
  @Test
  public void onePolicyExceptionInFlowErrorContinue() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, ERROR_CONTINUE_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorContinueRequest.get();
      assertThat(response.statusCode(), is(200));
      assertThat(response.asString(), is(FLOW_PAYLOAD + FLOW_ERROR_PAYLOAD + AFTER_PAYLOAD));
    });
  }

  @Description("One policy: error thrown in the flow which has defined an error handler with on-error-propagate")
  @Test
  public void onePolicyExceptionInFlowErrorPropagate() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, ERROR_PROPAGATE_API_KEY, config.getConfigData());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorPropagateRequest.get();
      assertThat(response.asString(), is(FLOW_PAYLOAD + FLOW_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, 500);
    });
  }

  @Description("Two policies: error thrown in the before part of the inner policy which does not have an error handler")
  @Test
  public void twoPoliciesErrorBeforeSecondNextAndNoErrorHandler() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());
    policyDefinition2 = policyDefinition(2, BEFORE_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY, DEFAULT_NO_ERROR_HANDLER_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.asString(), is(BEFORE_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, NONE);
    });
  }

  @Description("Two policies: error thrown in the before part of the inner policy which overrides status code and adds headers")
  @Test
  public void twoPoliciesErrorBeforeSecondNextAndSettingSCAndHeaders() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());
    policyDefinition2 = policyDefinition(2, BEFORE_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY,
                                         ON_ERROR_PROPAGATE_OVERRIDE_HEADERS_SC_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertVariableErrorStatusCode(response, 401);
      assertThat(response.asString(), is(BEFORE_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
    });
  }

  @Description("Two policies: error thrown in the before part of the inner policy which has an error handler with on-error-continue")
  @Test
  public void twoPoliciesErrorBeforeSecondNextAndErrorContinue() {

    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());
    policyDefinition2 = policyDefinition(2, BEFORE_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY, DEFAULT_ON_ERROR_CONTINUE_CONFIG_DATA);
    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.statusCode(), is(200));
      assertThat(response.asString(), is(BEFORE_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + AFTER_PAYLOAD));
      assertThat(response.header("testHeader"), nullValue());
    });
  }

  @Description("Two policies: error thrown in the before part of the inner policy which has an error handler with on-error-propagate")
  @Test
  public void twoPoliciesErrorBeforeSecondNextAndErrorPropagate() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());

    policyDefinition2 = policyDefinition(2, BEFORE_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY,
                                         DEFAULT_ON_ERROR_PROPAGATE_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      // Outside error-handler is never executed
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.asString(), is(BEFORE_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, NONE);
    });
  }

  @Description("Two policies: error thrown in the after part of the inner policy which does not have an error handler")
  @Test
  public void twoPoliciesErrorAfterSecondNextAndNoErrorHandler() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());

    policyDefinition2 = policyDefinition(2, AFTER_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY,
                                         DEFAULT_NO_ERROR_HANDLER_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.asString(), is(FLOW_PAYLOAD + AFTER_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, 200);
    });
  }

  @Description("Two policies: error thrown in the after part of the inner policy which has an error handler with on-error-continue")
  @Test
  public void twoPoliciesErrorAfterSecondNextAndErrorContinue() {

    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());
    policyDefinition2 = policyDefinition(2, AFTER_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY,
                                         DEFAULT_ON_ERROR_CONTINUE_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      // Outside error-handler is never executed
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.statusCode(), is(200));
      assertThat(response.asString(), is(FLOW_PAYLOAD + AFTER_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + AFTER_PAYLOAD));
      assertThat(response.header("testHeader"), nullValue());
    });
  }

  @Description("Two policies: error thrown in the after part of the inner policy which has an error handler with on-error-propagate")
  @Test
  public void twoPoliciesErrorAfterSecondNextAndErrorPropagate() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());
    policyDefinition2 = policyDefinition(2, AFTER_NEXT_TEMPLATE_ID, SET_PAYLOAD_API_KEY,
                                         DEFAULT_ON_ERROR_PROPAGATE_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.asString(),
                 is(FLOW_PAYLOAD + AFTER_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, 200);
    });
  }

  @Description("Two policies: error thrown in the error handler of the inner policy.")
  @Test
  public void twoPoliciesErrorInSecondErrorHandler() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, config.getConfigData());
    policyDefinition2 = policyDefinition(2, IN_ERROR_HANDLER_TEMPLATE_ID, SET_PAYLOAD_API_KEY,
                                         DEFAULT_ON_ERROR_CONTINUE_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();
      assertThat(response.asString(), is(BEFORE_PAYLOAD + POLICY_SECOND_ERROR_PAYLOAD + policyErrorHandlerPayload()));
      assertHeaderIncluded(response);
      assertVariableErrorStatusCode(response, NONE);
    });
  }

  @Description("Two policies: error thrown in the flow.")
  @Test
  public void twoPoliciesErrorInFlow() {
    policyDefinition = policyDefinition(1, NO_EXCEPTION_TEMPLATE_ID, ERROR_RESPONSE_API_KEY, config.getConfigData());
    policyDefinition2 =
        policyDefinition(2, NO_EXCEPTION_TEMPLATE_ID, ERROR_RESPONSE_API_KEY, DEFAULT_NO_ERROR_HANDLER_CONFIG_DATA);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    serialExecutor.execute(() -> {
      HttpResponse response = errorResponseRequest.get();
      assertVariableErrorStatusCode(response, 450);
      assertThat(response.asString(), is(FLOW_PAYLOAD + FLOW_ERROR_PAYLOAD + policyErrorHandlerPayload()));
    });
  }

  /**
   * Asserts that the status code is the expected one
   * 
   * @param response the HTTP Response
   * @param nextStatusCode the status code that was present when the execute-next element returned to the policy being tested,
   *        which can be from the flow or from another policy.
   */
  protected abstract void assertVariableErrorStatusCode(HttpResponse response, int nextStatusCode);

  /**
   * @return the value that the policy's error handler that is being tested is setting when executed
   */
  protected abstract String policyErrorHandlerPayload();

  /**
   * Asserts that a header was included when executed the error handler. This should be asserted only when the error-handler was
   * configured to add one
   */
  private void assertHeaderIncluded(HttpResponse response) {
    if (INCLUDE_HEADERS.equals(config.getHeadersMode())) {
      assertThat(response.header("testHeader"), is("testValue"));
    }
  }

}
