/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.error.source;

import static com.google.common.collect.ImmutableMap.of;
import static java.util.Collections.emptyMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import com.mulesoft.anypoint.tests.http.HttpResponse;

import io.qameta.allure.Description;
import org.junit.Test;

public class PolicySourceErrorHandlingScenariosNested extends PolicySourceErrorHandlingInfrastructure {

  @Description("One policy: Error thrown before executing the flow. The policy contains a nested error handler, and the exception is propagated through the entire exception chain.")
  @Test
  public void nestedErrorHandlerInPolicyAndExceptionBeforeFlow() {
    policyDefinition =
        policyDefinition(1, POLICY_WITH_NESTED_TRY_THROWS_EXCEPTION, SET_PAYLOAD_API_KEY, of("exceptionBefore", true));

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.asString(), is(BEFORE_PAYLOAD + "TheException"));
    });
  }

  @Description("One policy: Error thrown after executing the flow. The policy contains a nested error handler, and the exception is propagated through the entire exception chain.")
  @Test
  public void nestedErrorHandlerInPolicyAndExceptionAfterFlow() {
    policyDefinition =
        policyDefinition(1, POLICY_WITH_NESTED_TRY_THROWS_EXCEPTION, SET_PAYLOAD_API_KEY, of("exceptionBefore", false));

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = setPayloadRequest.get();

      assertThat(response.statusCode(), is(200));
      assertThat(response.asString(), is(FLOW_PAYLOAD + AFTER_PAYLOAD + "TheException"));
    });
  }

  @Description("One policy with nested error handler applied into a flow that throws an exception without any error handling in it.")
  @Test
  public void nestedErrorHandlerInPolicyAndFlowThrowsException() {
    policyDefinition = policyDefinition(1, POLICY_WITH_NESTED_TRY_NO_EXCEPTION, ERROR_API_KEY, emptyMap());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.asString(), is(EXCEPTION_MESSAGE + "TheException"));
    });
  }

  @Description("One policy with nested error handler applied into a flow that propagates the exception.")
  @Test
  public void nestedErrorHandlerInPolicyAndFlowPropagatesError() {
    policyDefinition =
        policyDefinition(1, POLICY_WITH_NESTED_TRY_NO_EXCEPTION, ERROR_PROPAGATE_API_KEY, emptyMap());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.asString(), is(EXCEPTION_MESSAGE + "TheException"));
    });
  }

  @Description("One policy with nested error handler applied into a flow that catches the exception.")
  @Test
  public void nestedErrorHandlerInPolicyAndFlowContinuesError() {
    policyDefinition =
        policyDefinition(1, POLICY_WITH_NESTED_TRY_NO_EXCEPTION, ERROR_CONTINUE_API_KEY, emptyMap());

    server.deployPolicy(policyDefinition);

    serialExecutor.execute(() -> {
      HttpResponse response = errorRequest.get();

      assertThat(response.statusCode(), is(500));
      assertThat(response.asString(), is(EXCEPTION_MESSAGE + "TheException"));
    });
  }
}
