/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.gatekeeper;

import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.BLOCKED;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.UNBLOCKED;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static java.util.Collections.emptyList;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;

import org.mule.runtime.api.healthcheck.ReadyStatus;

import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;
import com.mulesoft.mule.runtime.gw.model.contracts.repository.MapDBContractRepository;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStoreFiles;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class FlexibleGatekeeperTrackingFailureTestCase extends FlexibleGateKeeperTestCase {

  private final FakeGatewayInstallation installation = installation();

  @Rule
  public RuleChain chain = RuleChain.outerRule(httpPort)
      .around(httpPort2)
      .around(httpPort3)
      .around(apiPort)
      .around(platformUri)
      .around(fakeApiServerRule)
      .around(installation);

  @Override
  protected FakeGatewayServer getServer() {
    return installation.getServer();
  }

  @Before
  public void setUp() {
    fakeModel().setGetApiResourceStatus(INTERNAL_SERVER_ERROR.getStatusCode());

    this.storeFiles = new PolicyStoreFiles(getServer().getPoliciesDir());
    this.contractRepository = new MapDBContractRepository();
  }

  @After
  public void tearDown() {
    contractRepository.dispose();
    getServer().getDeploymentService()
        .getApplications()
        .forEach(application -> getServer().undeployApplication(application.getArtifactName()));
    getServer().resetDeploymentListener();
    fakeModel().clear();
  }

  @Test
  public void apiUnblockedWhenTrackingFails() {
    fakeModel().createApi(1L);

    getServer().deployApplications(appWithGoodPolicy);

    probe(() -> {
      HttpResponse response = request1.get();
      assertThat(response.statusCode(), is(UNBLOCKED));
      assertThat(response.asString(), is(POLICY_PAYLOAD));
      assertThat(trackingInfo(API_KEY).isFailedTracking(), is(true));
      assertTrafficAllowed(true);
    });
  }

  @Test
  public void apiBlockedWhenTrackingFailsAndPolicyDeploymentFails() {
    fakeModel().createApi(2L);

    getServer().deployApplications(appWithBadPolicy);

    getServer().deployApplications(witnessApp);

    probe(() -> {
      HttpResponse response = request2.get();
      assertThat(storeFiles.listPolicyDeploymentFailures(), hasSize(1));
      assertThat(response.statusCode(), is(BLOCKED));
      assertThat(trackingInfo(API_KEY_2).isFailedTracking(), is(true));
    });

    assertTrafficAllowed(false);
  }

  @Test
  public void apiUnblockedWhenTrackingFailsAndNoPoliciesStored() {
    fakeModel().createApi(3L);
    contractRepository.storeSlas(API_KEY_3, emptyList());

    getServer().deployApplications(appWithNoPolicy);

    probe(() -> {
      HttpResponse response = request3.get();
      assertThat(response.statusCode(), is(UNBLOCKED));
      assertThat(trackingInfo(API_KEY_3).isFailedTracking(), is(true));
      assertTrafficAllowed(true);
    });
  }

  @Test
  public void apiBlockedWhenTrackingFailsForFirstDeployment() {
    fakeModel().createApi(3L);

    getServer().deployApplications(appWithNoPolicy);

    getServer().deployApplications(witnessApp);

    probe(() -> {
      HttpResponse response = request3.get();
      assertThat(response.statusCode(), is(BLOCKED));
      assertThat(trackingInfo(API_KEY_3).isFailedTracking(), is(true));
    });

    assertTrafficAllowed(false);
    readyStatus(appWithNoPolicy.getName(), false,
                "API 3: Not Ready. Awaiting for successful API tracking.");
  }

  @Test
  public void apiNotUnblockedWhenApiDoesntExist() {

    getServer().deployApplications(appWithNoPolicy);

    getServer().deployApplications(witnessApp);

    probe(() -> {
      HttpResponse response = request3.get();
      assertThat(response.statusCode(), is(BLOCKED));
      assertThat(trackingInfo(API_KEY_3).isTracked(), is(false));
      assertThat(trackingInfo(API_KEY_3).isFailedTracking(), is(false));
    });

    assertTrafficAllowed(false);
  }

  private void readyStatus(String appName, boolean unblocked, String message) {
    ReadyStatus status = installation.getServer().getHealthCheck().getValidator(appName).ready();

    assertThat(status.isReady(), is(unblocked));
    assertThat(status.statusDescription().isPresent(), is(true));
    assertThat(status.statusDescription().get(), is(message));
  }

  private TrackingInfo trackingInfo(ApiKey apiKey) {
    return getServer().getApiService().get(apiKey).get().getTrackingInfo();
  }
}
