/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.gatekeeper;

import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.BLOCKED;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.UNBLOCKED;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyTemplateKey.from;
import static com.mulesoft.anypoint.tita.TestDependencies.contractsExtensionDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode.FLEXIBLE;
import static com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode.STRICT;
import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.BACKOFF;
import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.POLL_CLIENTS_FREQ;
import static java.util.Arrays.asList;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.ApacheHttpRequest;
import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tests.infrastructure.model.FakeApi;
import com.mulesoft.anypoint.tests.infrastructure.model.FakeApiClient;
import com.mulesoft.anypoint.tests.infrastructure.model.FakePolicy;
import com.mulesoft.anypoint.tests.infrastructure.rules.FakeApiServerRule;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.BackoffScheduler;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import com.google.common.collect.ImmutableMap;

import java.util.Collection;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

@RunWith(Parameterized.class)
public class GateKeeperPreemptiveContractsTestCase extends AbstractMuleTestCase {

  private static DynamicPort apiPort = new DynamicPort("apiPort");
  private static FakeApiServerRule fakeApiServerRule = new FakeApiServerRule(apiPort.getNumber());
  private static SystemProperty platformUri =
      new SystemProperty("anypoint.platform.base_uri", "http://localhost:" + apiPort.getNumber() + "/test");

  private static SystemProperty pollFreq = new SystemProperty(POLL_CLIENTS_FREQ, "3");
  private static SystemProperty backoff = new SystemProperty(BACKOFF, "false");

  private static DynamicPort httpPort = new DynamicPort("port");
  private static DynamicPort httpPort2 = new DynamicPort("port2");
  private static Artifact application = buildTestApplication(APP, "mule-config-single-api.xml");
  private static Artifact application2 = buildTestApplication(APP_2, "mule-config-single-different-api.xml");


  private final SystemProperty policyPayload2 = new SystemProperty("policyPayload2", POLICY_PAYLOAD_2);

  private static final FakeApiClient client = FakeApiClient.builder()
      .withId("clientId")
      .withSecret("clientSecret")
      .withName("clientName")
      .build();


  @ClassRule
  public static RuleChain serverChain = RuleChain
      .outerRule(apiPort)
      .around(fakeApiServerRule)
      .around(platformUri);

  @Rule
  public RuleChain chain = RuleChain
      .outerRule(httpPort)
      .around(httpPort2)
      .around(policyPayload2)
      .around(pollFreq)
      .around(backoff);

  protected FakeGatewayInstallation installation;

  private HttpRequest request = request(httpPort, "/api/main");


  private Mode mode;
  private TestDownloaderListener deploymentListener;

  @Parameters(name = "{0} GateKeeper")
  public static Collection<Object[]> data() {
    return asList(new Object[][] {
        {STRICT},
        {FLEXIBLE}
    });
  }

  public GateKeeperPreemptiveContractsTestCase(Mode mode) {
    this.mode = mode;
    this.installation = installation();
    this.chain = chain.around(installation);
  }

  @After
  public void tearDown() {
    installation.getServer().removeAllPoliciesAndContext();
    fakeModel().clear();
  }

  @Test
  public void apiContractsArePreemptivelyFetch() {

    //deploy aux application to finish gateway initialization.
    installation.getServer().deployApplications(application2);

    //Stop the clients poller to ensure that the preemptive fetch is the one that retrieved the contract.
    new Inspector(installation.getServer().getPollersManager())
        .<BackoffScheduler>read("schedulerClients")
        .dispose();

    deploymentListener = new TestDownloaderListener();
    installation.getServer().getPolicySetDeploymentService().addPolicyDeploymentListener(deploymentListener);
    installation.getServer().getApiService().addApiContractsListener(deploymentListener);

    FakeApi api = fakeModel().createApi(1L);
    api.addPolicy(new FakePolicy(1, from(POLICY_TEMPLATE_KEY), 1, ImmutableMap.of("apiId", api.getId().toString())));
    fakeModel().setClients(api.getId(), client);

    installation.getServer().deployApplications(application);

    deploymentListener
        .waitPoliciesUpdated()
        .waitContractsPrefetch()
        .waitContractsUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertThat(installation.getServer().getHealthCheck().isApplicationTrafficAllowed(application.getName()), is(true));

    GatekeeperStatusInspector inspector = new GatekeeperStatusInspector(
                                                                        installation.getServer().getApiService()
                                                                            .get(new ApiKey(api.getId())).get()
                                                                            .getImplementation().gatekeeperStatus());

    assertThat(inspector.policiesApplied(), is(true));
    assertThat(inspector.requiresContracts(), is(true));
    assertThat(inspector.contractsAvailable(), is(true));

  }

  @Test
  public void apiContractsArePreemptivelyFetchFails() {

    //deploy aux application to finish gateway initialization.
    installation.getServer().deployApplications(application2);

    deploymentListener = new TestDownloaderListener();
    installation.getServer().getPolicySetDeploymentService().addPolicyDeploymentListener(deploymentListener);
    installation.getServer().getApiService().addApiContractsListener(deploymentListener);

    FakeApi api = fakeModel().createApi(1L);
    api.addPolicy(new FakePolicy(1, from(POLICY_TEMPLATE_KEY), 1, ImmutableMap.of("apiId", api.getId().toString())));
    fakeModel().setClients(api.getId(), client);

    fakeModel().setContractsStatus(503);

    installation.getServer().deployApplications(application);

    deploymentListener
        .waitPoliciesUpdated()
        .waitContractsPrefetch();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertThat(installation.getServer().getHealthCheck().isApplicationTrafficAllowed(application.getName()), is(false));
    GatekeeperStatusInspector inspector = new GatekeeperStatusInspector(
                                                                        installation.getServer().getApiService()
                                                                            .get(new ApiKey(api.getId())).get()
                                                                            .getImplementation().gatekeeperStatus());

    assertThat(inspector.policiesApplied(), is(true));
    assertThat(inspector.requiresContracts(), is(true));
    assertThat(inspector.contractsAvailable(), is(false));


    fakeModel().setContractsStatus(200);

    deploymentListener
        .waitContractsUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertThat(installation.getServer().getHealthCheck().isApplicationTrafficAllowed(application.getName()), is(true));

    inspector = new GatekeeperStatusInspector(
                                              installation.getServer().getApiService()
                                                  .get(new ApiKey(api.getId())).get()
                                                  .getImplementation().gatekeeperStatus());

    assertThat(inspector.policiesApplied(), is(true));
    assertThat(inspector.requiresContracts(), is(true));
    assertThat(inspector.contractsAvailable(), is(true));

  }

  private HttpRequest request(DynamicPort port, String s) {
    return ApacheHttpRequest.request(port, s).withTimeout(10000);
  }

  private FakeGatewayInstallation installation() {
    FakeGatewayInstallation.Builder builder = builder()
        .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY, "policies/policy-deployment-with-contracts.xml",
                                                     contractsExtensionDependency()));

    if (mode.equals(FLEXIBLE)) {
      builder.flexibleGateKeeper();
    }

    return builder.build();
  }

}
