/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.gatekeeper;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.BLOCKED;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.INVALID_POLICY_TEMPLATE_ID;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.UNBLOCKED;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.contractsPolicyDefinition;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.invalidDefinition;
import static com.mulesoft.anypoint.test.policy.gatekeeper.GateKeeperTestData.policyDefinition;
import static com.mulesoft.anypoint.tests.CountProber.descriptiveCountProber;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_3;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.TestDependencies.contractsExtensionDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode.FLEXIBLE;
import static com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode.STRICT;
import static com.mulesoft.mule.runtime.gw.model.PolicySet.PolicySetOrigin.PLATFORM;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static org.apache.http.HttpHeaders.CONTENT_LENGTH;
import static org.apache.http.HttpHeaders.CONTENT_TYPE;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.isEmptyString;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;

import org.mule.runtime.api.healthcheck.ReadyStatus;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.ApacheHttpRequest;
import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.ApiTrackingInfo;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import io.qameta.allure.Issue;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

@RunWith(Parameterized.class)
public class GateKeeperStandaloneTestCase extends AbstractMuleTestCase {

  private static final String SOME_PAYLOAD = "SomethinSomethin";
  private static final String SOME_HEADER = "someHeader";
  private static final String SOME_HEADER_VALUE = "someValue";

  private static Artifact application = buildTestApplication(APP, "mule-config-single-api.xml");
  private static Artifact application2 = buildTestApplication(APP_2, "mule-config-multiple-flows.xml");

  private final SystemProperty policyPayload2 = new SystemProperty("policyPayload2", POLICY_PAYLOAD_2);
  private final SystemProperty blockApi = new SystemProperty("anypoint.platform.policies.on_api_deleted", "block_api");

  private static DynamicPort httpPort = new DynamicPort("port");
  private static DynamicPort httpPort2 = new DynamicPort("port2");

  @Rule
  public RuleChain chain = RuleChain.outerRule(httpPort)
      .around(httpPort2)
      .around(policyPayload2)
      .around(blockApi);

  protected FakeGatewayInstallation installation;

  private HttpRequest request = request(httpPort, "/api/main");

  private PolicySetDeploymentService policySetDeploymentService;

  private PolicyDeploymentService policyDeploymentService;

  private Mode mode;
  private ApiTrackingService apiTrackingService;
  private TestDownloaderListener deploymentListener;

  @Parameters(name = "{0} GateKeeper")
  public static Collection<Object[]> data() {
    return asList(new Object[][] {
        {STRICT},
        {FLEXIBLE}
    });
  }

  public GateKeeperStandaloneTestCase(Mode mode) {
    this.mode = mode;
    this.installation = installation();
    this.chain = chain.around(installation);
  }

  @Before
  public void setUp() {
    policySetDeploymentService = installation.getServer().getPolicySetDeploymentService();
    policyDeploymentService = installation.getServer().getPolicyDeploymentService();
    apiTrackingService = installation.getServer().getApiTrackingService();

    deploymentListener = new TestDownloaderListener();
    policySetDeploymentService.addPolicyDeploymentListener(deploymentListener);
    installation.getServer().getApiService().addApiContractsListener(deploymentListener);
  }

  @After
  public void tearDown() {
    installation.getServer().removeAllPoliciesAndContext();
  }

  @Test
  public void apiIsBlocked() {
    HttpResponse response = request(httpPort, "/api/main")
        .withPayload(SOME_PAYLOAD)
        .withHeader(SOME_HEADER, SOME_HEADER_VALUE)
        .post();

    assertThat(response.statusCode(), is(BLOCKED));
    // Payload nor headers are echoed
    assertThat(response.asString(), isEmptyString());
    assertThat(response.allHeaders(SOME_HEADER), is(empty()));
    assertThat(response.header(CONTENT_TYPE), is("text/plain; charset=UTF-8"));
    assertThat(response.header(CONTENT_LENGTH), is("0"));

    assertInstanceTrafficAllowed(false);

    readyStatus(false, "API 1: Not Ready. Awaiting for successful API tracking.");
  }

  @Test
  public void onlyFlowsWithApisAreBlocked() {
    installation.getServer().deployApplications(application2);
    HttpRequest requestApi1 = request(httpPort2, "/api/api1");
    HttpRequest requestApi2 = request(httpPort2, "/api/api2");
    HttpRequest requestNoApi = request(httpPort2, "/api/no-api");

    assertThat(requestApi1.get().statusCode(), is(BLOCKED));
    assertThat(requestApi2.get().statusCode(), is(BLOCKED));
    assertThat(requestNoApi.get().statusCode(), is(UNBLOCKED));

    assertInstanceTrafficAllowed(false);
  }

  @Test
  public void anotherPolicyWithSameOrderThanGateKeeperPolicyIsNotExecuted() {
    installation.getServer().deployPolicy(policyDefinition);

    HttpResponse response = request.get();
    assertThat(response.statusCode(), is(BLOCKED));
    assertThat(response.asString(), isEmptyString());

    assertInstanceTrafficAllowed(false);
  }

  @Test
  public void apiIsUnblockedAfterSuccessfulPollingCycle() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    HttpResponse response = request.get();
    assertThat(response.statusCode(), is(UNBLOCKED));
    assertThat(response.asString(), is(POLICY_PAYLOAD));
    assertInstanceTrafficAllowed(true);
  }

  @Test
  public void apiIsUnblockedAfterEmptyPollingCycle() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertInstanceTrafficAllowed(true);
    readyStatus(true, "API 1: Ready.");
  }

  @Test
  public void apiIsUnblockedEvenWithFailedOffline() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policyDeploymentService
        .newPolicy(installation.getServer().getPolicyFactory()
            .createFromPolicyDefinition(new OfflinePolicyDefinition(POLICY_ID_2, INVALID_POLICY_TEMPLATE_ID, API_KEY, null, 1,
                                                                    emptyMap())));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertInstanceTrafficAllowed(true);
  }

  @Test
  public void apiIsNotUnblockedAfterFailedPollingCycle() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService
        .policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition, invalidDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertInstanceTrafficAllowed(false);
    readyStatus(false, "API 1: Not Ready. Error applying policies.");
  }

  @Test
  public void apiIsUnblockedAfterSecondPollingCycles() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService
        .policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition, invalidDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    HttpResponse response = request.get();
    assertThat(response.statusCode(), is(UNBLOCKED));
    assertThat(response.asString(), is(POLICY_PAYLOAD));
    assertInstanceTrafficAllowed(true);
  }

  @Test
  public void apiIsBlockedAgainWhenApiIsUntracked() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    HttpResponse response = request.get();
    assertThat(response.statusCode(), is(UNBLOCKED));
    assertThat(response.asString(), is(POLICY_PAYLOAD));
    assertInstanceTrafficAllowed(true);

    apiTrackingService.apiUntracked(API_KEY);

    deploymentListener.waitPoliciesRemoved();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertInstanceTrafficAllowed(false);
    readyStatus(false, "API 1: Not Ready. API not found in the API Platform.");
  }

  @Test
  public void twoApisOneBlockedAndOneUnblocked() {
    installation.getServer().deployApplications(application2);
    HttpRequest requestApiApp1 = request;
    HttpRequest requestApi1App2 = request(httpPort2, "/api/api1");
    HttpRequest requestApi2App2 = request(httpPort2, "/api/api2");

    assertThat(requestApiApp1.get().statusCode(), is(BLOCKED));
    assertThat(requestApi1App2.get().statusCode(), is(BLOCKED));
    assertThat(requestApi2App2.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));
    policySetDeploymentService.policiesForApi(API_KEY_2, new PolicySet(newArrayList(), PLATFORM));

    probe(() -> {
      HttpResponse responseApi1 = requestApiApp1.get();
      assertThat(responseApi1.statusCode(), is(UNBLOCKED));
      assertThat(responseApi1.asString(), is(POLICY_PAYLOAD));

      HttpResponse responseApi2 = requestApi1App2.get();
      assertThat(responseApi2.statusCode(), is(UNBLOCKED));
    });

    assertThat(requestApi2App2.get().statusCode(), is(BLOCKED));

    assertInstanceTrafficAllowed(false);
    assertApplicationTrafficAllowed(application.getName(), true);
    assertApplicationTrafficAllowed(application2.getName(), false);

    readyStatus(application2.getName(), false,
                "Gatekeeper is blocking the app due to:\n" +
                    "API 2: Ready.\n" +
                    "API 3: Not Ready. Awaiting for successful API tracking.");
  }

  @Test
  public void apiIsBlockedOnAppRedeploy() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(), PLATFORM));

    probe(() -> {
      assertThat(request.get().statusCode(), is(UNBLOCKED));
      assertInstanceTrafficAllowed(true);
    });

    installation.getServer().getDeploymentService().redeploy(application.getName());

    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(), PLATFORM));

    probe(() -> {
      assertThat(request.get().statusCode(), is(UNBLOCKED));
      assertInstanceTrafficAllowed(true);
    });
  }

  @Test
  public void appStopAndStartKeepsGateKeeperUnblocked() {
    assertThat(request.get().statusCode(), is(BLOCKED));

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    probe(() -> {
      assertThat(request.get().statusCode(), is(UNBLOCKED));
      assertInstanceTrafficAllowed(true);
    });

    String appName = application.getName();

    installation.getServer().getDeploymentService().findApplication(appName).stop();
    installation.getServer().getDeploymentService().findApplication(appName).start();

    probe(() -> {
      assertThat(request.get().statusCode(), is(UNBLOCKED));
      assertInstanceTrafficAllowed(true);
    });
  }

  @Test
  public void apisRequiringContractsRemainBlockedAfterPolicyApplied() {
    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener
        .waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertInstanceTrafficAllowed(false);
    readyStatus(false, "API 1: Not Ready. Waiting for API's client applications to be successfully retrieved.");
  }

  @Test
  public void redeployPolicyFailureAfterPolicySuccessBlocks() {

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));
    deploymentListener.waitPoliciesUpdated();

    policySetDeploymentService
        .policiesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition, invalidDefinition), PLATFORM));
    deploymentListener.waitPoliciesUpdated();

    apiTrackingService.clientsPolling(API_KEY, platformContractAdaptersList());
    deploymentListener.waitContractsUpdated();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertInstanceTrafficAllowed(false);
    readyStatus(false, "API 1: Not Ready. Error applying policies.");
  }

  @Test
  public void redeployPolicyFailureInUnblockedApi() {

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));
    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));

    policySetDeploymentService
        .policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition, invalidDefinition), PLATFORM));
    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertInstanceTrafficAllowed(true);
    readyStatus(true, "API 1: Ready.");
  }


  @Test
  public void apisRequiringContractsRemainBlockedAfterPolicyAppliedAndSlasAvailable() {
    apiTrackingService.apiTracked(API_KEY, trackingInfo(), new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM),
                                  emptyList());

    deploymentListener
        .waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertInstanceTrafficAllowed(false);
    readyStatus(false, "API 1: Not Ready. Waiting for API's client applications to be successfully retrieved.");
  }

  @Test
  public void undeployPolicyWithContracts() {
    policySetDeploymentService.policiesForApi(API_KEY,
                                              new PolicySet(newArrayList(policyDefinition, contractsPolicyDefinition), PLATFORM));

    deploymentListener
        .waitPoliciesUpdated();

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    deploymentListener
        .waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertInstanceTrafficAllowed(true);
    readyStatus(true, "API 1: Ready.");
  }


  @Test
  public void apisWithContractsAreUnblockedWhenContractsPolledAfterPolicy() {
    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertInstanceTrafficAllowed(false);

    apiTrackingService.clientsPolling(API_KEY, platformContractAdaptersList());

    deploymentListener.waitContractsUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertInstanceTrafficAllowed(true);
    assertThat(request.get().asString(), is(POLICY_PAYLOAD_2));
  }

  @Test
  public void apisWithContractsAreUnblockedWhenContractsPolledBeforePolicy() {

    apiTrackingService.clientsPolling(API_KEY, platformContractAdaptersList());

    deploymentListener.waitContractsUpdated();

    assertThat(request.get().statusCode(), is(BLOCKED));
    assertInstanceTrafficAllowed(false);

    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertInstanceTrafficAllowed(true);
    assertThat(request.get().asString(), is(POLICY_PAYLOAD_2));
  }

  @Test
  public void apisWithContractsAreUnblockedWhenEmptyContractsPolled() {

    apiTrackingService.clientsPolling(API_KEY, emptyList());
    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener
        .waitContractsUpdated()
        .waitPoliciesUpdated();

    assertThat(request.get().statusCode(), is(UNBLOCKED));
    assertInstanceTrafficAllowed(true);
    assertThat(request.get().asString(), is(POLICY_PAYLOAD_2));
  }

  @Test
  @Issue("MULE-19191")
  public void unblockedApiIsUnblockedAgainAfterRuntimeRestartOnlyInFlexible() {
    policySetDeploymentService.policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));
    deploymentListener.waitPoliciesUpdated();
    HttpResponse response = request.get();
    assertThat(response.statusCode(), is(UNBLOCKED));
    assertThat(response.asString(), is(POLICY_PAYLOAD));
    assertInstanceTrafficAllowed(true);

    installation.restart();

    response = request.get();
    if (mode.equals(STRICT)) {
      assertThat(response.statusCode(), is(BLOCKED));
      assertThat(response.asString(), isEmptyString());
      assertInstanceTrafficAllowed(false);
      readyStatus(false, "API 1: Not Ready. Awaiting for successful API tracking.");
    } else {
      assertThat(request.get().statusCode(), is(UNBLOCKED));
      assertThat(response.asString(), is(POLICY_PAYLOAD));
      assertInstanceTrafficAllowed(true);
    }
  }

  private ApiTrackingInfo trackingInfo() {
    return new ApiTrackingInfo(API_ID, "instanceName", "groupId", "assetId", "version", "orgId",
                               "envId", "http", API_ID.intValue(), "entityTag",
                               "exchangeAssetName", "productVersion");
  }

  private List<PlatformContractAdapter> platformContractAdaptersList() {
    List<PlatformContractAdapter> platformContractAdapters = new LinkedList<>();

    platformContractAdapters.add(new PlatformContractAdapter(
                                                             new ApiClientDto("clientId", "clientSecret", "name", "orgId", "1",
                                                                              "apiId",
                                                                              "contractId", emptySet())));

    return platformContractAdapters;
  }

  private HttpRequest request(DynamicPort port, String s) {
    return ApacheHttpRequest.request(port, s).withTimeout(10000);
  }

  private FakeGatewayInstallation installation() {
    FakeGatewayInstallation.Builder builder = builder()
        .withApplications(application)
        .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY, "templates/policy-template.xml"),
                             buildTestPolicyTemplate(INVALID_POLICY_TEMPLATE_ID, "templates/empty-template.xml"),
                             buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_3, "policies/policy-deployment-with-contracts.xml",
                                                     contractsExtensionDependency()))
        .offline();

    if (mode.equals(FLEXIBLE)) {
      builder.flexibleGateKeeper();
    }

    return builder.build();
  }

  private void readyStatus(boolean unblocked, String message) {
    readyStatus(application.getName(), unblocked, message);
  }

  private void readyStatus(String appName, boolean unblocked, String message) {
    ReadyStatus status = installation.getServer().getHealthCheck().getValidator(appName).ready();

    assertThat(status.isReady(), is(unblocked));
    assertThat(status.statusDescription().isPresent(), is(true));
    assertThat(status.statusDescription().get(), is(message));
  }

  private void assertInstanceTrafficAllowed(boolean trafficAllowed) {
    assertThat(installation.getServer().getHealthCheck().isInstanceTrafficAllowed(), is(trafficAllowed));
  }

  private void assertApplicationTrafficAllowed(String applicationName, boolean trafficAllowed) {
    assertThat(installation.getServer().getHealthCheck().isApplicationTrafficAllowed(applicationName),
               is(trafficAllowed));
    assertThat(installation.getServer().getHealthCheck().getValidator(applicationName).ready().isReady(), is(trafficAllowed));
  }

  private void probe(Runnable runnable) {
    descriptiveCountProber(50, runnable);
  }

}
