/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.hdp.healthcheck;

import static com.google.common.collect.ImmutableMap.of;
import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_3;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_3;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.TestDependencies.contractsExtensionDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static com.mulesoft.mule.runtime.gw.model.PolicySet.PolicySetOrigin.PLATFORM;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.test.policy.gatekeeper.TestDownloaderListener;
import com.mulesoft.anypoint.tests.http.ApacheHttpRequest;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.gw.api.agent.HealthCheck;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.ApiTrackingInfo;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.model.hdp.ApiRegistry;

import java.io.File;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

public class HdpApiHealthCheckTestCase extends AbstractMuleTestCase {


  private static final String APP_NAME = "app-0.0.1-mule-application";
  private static final String APP_PAYLOAD = "ACK";

  private static PolicyDefinition contractsPolicyDefinition =
      new PolicyDefinition(POLICY_ID_3, POLICY_TEMPLATE_KEY_3, API_KEY, null, 1,
                           new PolicyConfiguration(of("apiId", API_KEY.id().toString())));



  private Artifact application = buildTestApplication(APP, "mule-config-hdp-proxy.xml");
  private ApiService apiService;

  @ClassRule
  public static DynamicPort httpPort = new DynamicPort("port");

  @ClassRule
  public static final SystemProperty policyPayload2 = new SystemProperty("policyPayload2", POLICY_PAYLOAD_2);

  @Rule
  public FakeGatewayInstallation installation = builder()
      .withApplications(application)
      .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_2, "templates/policy-template-response-headers.xml"),
                           buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_3, "policies/policy-deployment-with-contracts.xml",
                                                   contractsExtensionDependency()))
      .build();

  private final ApiRegistry registry = ApiRegistry.fromFile(APP_NAME,
                                                            new File(getClass().getResource("/hdp-api-registry/registry.json")
                                                                .getPath()));

  private TestDownloaderListener deploymentListener;
  private HealthCheck healtcheck;
  private ApiTrackingService apiTrackingService;

  @Before
  public void setUp() {
    apiService = installation.getServer().getApiService();
    apiTrackingService = installation.getServer().getApiTrackingService();

    deploymentListener = new TestDownloaderListener();
    installation.getServer().getPolicySetDeploymentService().addPolicyDeploymentListener(deploymentListener);
    installation.getServer().getApiService().addApiContractsListener(deploymentListener);
    healtcheck = installation.getServer().getHealthCheck();
  }

  @Test
  public void apiStartsBlockedAndGetUnblocked() {
    apiService.updateHdpApis(registry);
    assertBlocked(API_KEY);

    PolicyDefinition policyDefinition = new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY_2, API_KEY, emptyList(), 2,
                                                             new PolicyConfiguration(of("headerName", "myHeader", "headerValue",
                                                                                        "myHeaderValue")));

    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertUnblocked(API_KEY);
    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(true));
    assertThat(requestToService("reviews").get().header("myHeader"), is("myHeaderValue"));
  }

  @Test
  public void registryWithTwoApis() {
    ApiRegistry registry = ApiRegistry.fromFile(APP_NAME,
                                                new File(getClass().getResource("/hdp-api-registry/registry-two-apis.json")
                                                    .getPath()));

    apiService.updateHdpApis(registry);
    assertBlocked(API_KEY, API_KEY_2);

    PolicyDefinition policyDefinition = new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY_2, API_KEY, emptyList(), 2,
                                                             new PolicyConfiguration(of("headerName", "myHeader", "headerValue",
                                                                                        "myHeaderValue")));

    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertUnblocked(API_KEY);
    assertBlocked(API_KEY_2);
    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(false));

    assertThat(requestToService("reviews").get().header("myHeader"), is("myHeaderValue"));


    policyDefinition = new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY_2, API_KEY_2, emptyList(), 2,
                                            new PolicyConfiguration(of("headerName", "myHeader", "headerValue",
                                                                       "myHeaderValue2")));

    installation.getServer().deployPoliciesForApi(API_KEY_2, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertUnblocked(API_KEY_2);
    assertThat(requestToService("reviews2").get().header("myHeader"), is("myHeaderValue2"));

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(true));

  }

  @Test
  public void apisWithSlaAreBlocked() {
    apiService.updateHdpApis(registry);

    apiTrackingService.apiTracked(API_KEY, trackingInfo(), new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM),
                                  emptyList());
    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener
        .waitPoliciesUpdated();

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(false));
  }

  @Test
  public void apisWithContractsAreBlocked() {
    apiService.updateHdpApis(registry);

    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener
        .waitPoliciesUpdated();

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(false));
  }

  @Test
  public void apisWithContractsAreUnblockedWhenContractsPolledAfterPolicy() {
    apiService.updateHdpApis(registry);

    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(false));

    apiTrackingService.clientsPolling(API_KEY, platformContractAdapterList());

    deploymentListener.waitContractsUpdated();

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(true));
    assertThat(requestToService("reviews").get().asString(), is(APP_PAYLOAD + POLICY_PAYLOAD_2));
  }

  @Test
  public void apisWithContractsAreUnblockedWhenContractsPolledBeforePolicy() {
    apiService.updateHdpApis(registry);

    apiTrackingService.clientsPolling(API_KEY, platformContractAdapterList());

    deploymentListener.waitContractsUpdated();

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(false));

    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener.waitPoliciesUpdated();

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(true));
    assertThat(requestToService("reviews").get().asString(), is(APP_PAYLOAD + POLICY_PAYLOAD_2));
  }

  @Test
  public void apisWithContractsAreUnblockedWhenEmptyContractsPolled() {
    apiService.updateHdpApis(registry);

    apiTrackingService.clientsPolling(API_KEY, emptyList());
    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(contractsPolicyDefinition), PLATFORM));

    deploymentListener
        .waitContractsUpdated()
        .waitPoliciesUpdated();

    assertThat(healtcheck.isApplicationTrafficAllowed(APP_NAME), is(true));
    assertThat(requestToService("reviews").get().asString(), is(APP_PAYLOAD + POLICY_PAYLOAD_2));
  }

  private List<PlatformContractAdapter> platformContractAdapterList() {
    List<PlatformContractAdapter> platformContractAdapters = new LinkedList<>();

    platformContractAdapters.add(new PlatformContractAdapter(
                                                             new ApiClientDto("clientId", "clientSecret", "name", "orgId", "1",
                                                                              "apiId",
                                                                              "contractId", emptySet())));

    return platformContractAdapters;
  }

  private ApiTrackingInfo trackingInfo() {
    return new ApiTrackingInfo(API_ID, "instanceName", "groupId", "assetId", "version", "orgId",
                               "envId", "http", API_ID.intValue(), "entityTag",
                               "exchangeAssetName", "productVersion");
  }

  private ApacheHttpRequest requestToService(String service) {
    return request(httpPort, "/api/main/").withHeader("X-MULE-HDP-SERVICE", service);
  }


  void assertBlocked(ApiKey... keys) {
    Arrays.stream(keys).forEach(key -> assertThat(apiService
        .getApis()
        .stream()
        .filter(api -> api.getKey().id().equals(key.id()))
        .allMatch(api -> api.getImplementation().gatekeeperStatus().isBlocked()), is(true)));
  }

  void assertUnblocked(ApiKey... keys) {
    Arrays.stream(keys).forEach(key -> assertThat(apiService
        .getApis()
        .stream()
        .filter(api -> api.getKey().id().equals(key.id()))
        .noneMatch(api -> api.getImplementation().gatekeeperStatus().isBlocked()), is(true)));
  }
}
