/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.offline;

import static com.google.common.collect.ImmutableMap.of;
import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.CountProber.descriptiveCountProber;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getOfflinePoliciesFolder;
import static com.mulesoft.mule.runtime.gw.model.PolicySet.PolicySetOrigin.PLATFORM;
import static com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore.POLICY_DEFINITION_JSON_FILE_NAME;
import static java.util.Collections.emptyList;
import static org.apache.commons.io.FileUtils.cleanDirectory;
import static org.apache.commons.io.FileUtils.copyFile;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.isEmptyString;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore;
import com.mulesoft.mule.runtime.gw.policies.store.EncryptedPropertiesSerializer;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class OfflinePolicyTestCase extends AbstractMuleTestCase {

  private static DynamicPort httpPort = new DynamicPort("port");
  private static DynamicPort httpPort2 = new DynamicPort("port2");

  private static Artifact application = buildTestApplication(APP, "mule-config-single-api.xml");

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(application)
          .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY, "templates/policy-template.xml"),
                               buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_2, "templates/policy-template-response-headers.xml"))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain chain = RuleChain
      .outerRule(httpPort)
      .around(httpPort2)
      .around(installation);

  private HttpRequest request = request(httpPort, "/api/main");
  private HttpRequest request2 = request(httpPort2, "/api/main");

  private PolicyStore policyStore = new DefaultPolicyStore(new EncryptedPropertiesSerializer());

  @After
  public void tearDown() throws IOException, InterruptedException {
    installation.getServer().removeAllPoliciesAndContext();
    cleanDirectory(getOfflinePoliciesFolder());

    // Let a polling interval pass by for the offline directory watcher
    Thread.sleep(100);
  }

  @Test
  public void offlinePolicyIsApplied() throws IOException, URISyntaxException {
    copyFile(definitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));
  }

  @Test
  public void offlineAndOnlinePolicyTogether() throws IOException, URISyntaxException {
    copyFile(definitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));

    PolicyDefinition policyDefinition = new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY_2, API_KEY, emptyList(), 2,
                                                             new PolicyConfiguration(of("headerName", "myHeader", "headerValue",
                                                                                        "myHeaderValue")));
    installation.getServer().deployPolicy(policyDefinition);

    HttpResponse response = request.get();
    assertThat(response.asString(), is(POLICY_PAYLOAD));
    assertThat(response.header("myHeader"), is("myHeaderValue"));
  }

  @Test
  public void offlineAndOnlinePolicySetDeployment() throws IOException, URISyntaxException {
    copyFile(definitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));

    PolicyDefinition policyDefinition = new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY_2, API_KEY, emptyList(), 2,
                                                             new PolicyConfiguration(of("headerName", "myHeader", "headerValue",
                                                                                        "myHeaderValue")));
    installation.getServer().deployPoliciesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition), PLATFORM));

    descriptiveCountProber(50, () -> {
      HttpResponse response = request.get();
      assertThat(response.asString(), is(POLICY_PAYLOAD));
      assertThat(response.header("myHeader"), is("myHeaderValue"));
    });
  }

  @Test
  public void offlinePolicyIsUpdated() throws IOException, URISyntaxException {
    copyFile(definitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));

    copyFile(updatedDefinitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD_2)));
  }

  @Test
  public void offlinePolicyIsRemoved() throws IOException, URISyntaxException {
    File definitionFile = new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME);

    copyFile(definitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));

    definitionFile.delete();

    check(() -> assertThat(request.get().asString(), isEmptyString()));
  }

  @Test
  public void appRedeploymentKeepsOfflinePolicy() throws IOException, URISyntaxException {
    copyFile(definitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));

    installation.getServer().resetDeploymentListener();
    installation.getServer().getDeploymentService().redeploy(application.getName());

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));
  }

  @Test
  public void offlinePolicyWithMultipleApisWithSecondApiJoiningAfterPolicyIsDeployed() throws IOException, URISyntaxException {
    Artifact secondApp = buildTestApplication(APP_2, "mule-config-single-different-api.xml");
    copyFile(multipleApisDefinitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    check(() -> assertThat(request.get().asString(), is(POLICY_PAYLOAD)));

    installation.getServer().deployApplications(secondApp);

    check(() -> assertThat(request2.get().asString(), is(POLICY_PAYLOAD)));

    assertThat(policyStore.offlinePolicies(), hasSize(1));
    assertThat(policyStore.load(), hasSize(1));
  }

  private void check(Runnable asserter) {
    descriptiveCountProber(100, asserter);
  }

  private File definitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/full-definition.json").toURI());
  }

  private File updatedDefinitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/updated-definition.json").toURI());
  }

  private File multipleApisDefinitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/offline-multiple-apis-definition.json").toURI());
  }

}
