/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.scope;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.GROUP_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.TestDependencies.testAssertionsDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

@RunWith(Parameterized.class)
public class PolicyScopeVariablesTestCase extends PolicyScopeTestCase {

  private static final PolicyTemplateKey OPERATION_VARS_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "operationVars", "0.1.0");
  private static final PolicyTemplateKey SOURCE_VARS_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "sourceVars", "0.1.0");
  private static final PolicyTemplateKey OPERATION_AND_SOURCE_VARS_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "opAndSourceVars", "0.1.0");

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(app1, app2, app3)
          .withPolicyTemplates(buildTestPolicyTemplate(OPERATION_VARS_TEMPLATE_ID,
                                                       "policies/policy-operation-scope-variables-policy.xml",
                                                       testAssertionsDependency()),
                               buildTestPolicyTemplate(SOURCE_VARS_TEMPLATE_ID,
                                                       "policies/policy-source-scope-variables-policy.xml",
                                                       testAssertionsDependency()),
                               buildTestPolicyTemplate(OPERATION_AND_SOURCE_VARS_TEMPLATE_ID,
                                                       "policies/policy-operation-and-source-scope-variables-policy.xml",
                                                       testAssertionsDependency()))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(portApp1)
      .around(portApp2)
      .around(portApp3)
      .around(backend)
      .around(installation);

  private Boolean propagateMessageTransformations;

  private HttpRequest request;

  private FakeGatewayServer server = installation.getServer();

  public PolicyScopeVariablesTestCase(Boolean propagateMessageTransformations, DynamicPort port, ApplicationJar application) {
    this.propagateMessageTransformations = propagateMessageTransformations;
    this.request = request(port, "/api/variables");
    this.apiFinder = new ApiFinder(application.getAppConfig());
  }

  @Parameters(name = "{index}: Propagating message modification: {0}")
  public static Object[][] parameters() {
    return new Object[][] {
        {false, portApp1, app1},
        {true, portApp1, app1},
        {false, portApp2, app2},
        {true, portApp2, app2},
        {true, portApp3, app3},
        {false, portApp3, app3}
    };
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void varsHaveLocalScopeInOperationPolicies() {
    PolicyDefinition policyDefinition =
        policyDefinition(OPERATION_VARS_TEMPLATE_ID, getVariablesApiKey(), propagateMessageTransformations);
    policyDefinition.getConfigurationData().getConfiguration().put("policyId", "p1");
    policyDefinition.getConfigurationData().getConfiguration().put("otherId", "p2");

    PolicyDefinition policyDefinition2 =
        secondPolicy(OPERATION_VARS_TEMPLATE_ID, getVariablesApiKey(), propagateMessageTransformations);
    policyDefinition2.getConfigurationData().getConfiguration().put("policyId", "p2");
    policyDefinition2.getConfigurationData().getConfiguration().put("otherId", "p1");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = request.get();

    assertThat(response.asString(), response.statusCode(), is(200));
    assertThat(backendReceivedBody(), is(POLICY_PAYLOAD));
    assertThat(response.header("flow-variable"), is("value"));
  }

  @Test
  public void varsHaveLocalScopeInSourcePolicies() {
    PolicyDefinition policyDefinition =
        policyDefinition(SOURCE_VARS_TEMPLATE_ID, getVariablesApiKey(), propagateMessageTransformations);
    policyDefinition.getConfigurationData().getConfiguration().put("policyId", "p1");
    policyDefinition.getConfigurationData().getConfiguration().put("otherId", "p2");
    policyDefinition.getConfigurationData().getConfiguration().put("payload", POLICY_PAYLOAD);

    PolicyDefinition policyDefinition2 =
        secondPolicy(SOURCE_VARS_TEMPLATE_ID, getVariablesApiKey(), propagateMessageTransformations);
    policyDefinition2.getConfigurationData().getConfiguration().put("policyId", "p2");
    policyDefinition2.getConfigurationData().getConfiguration().put("otherId", "p1");
    policyDefinition2.getConfigurationData().getConfiguration().put("payload", POLICY_PAYLOAD_2);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = request.get();

    assertThat(response.asString(), response.statusCode(), is(200));
    assertThat(response.asString(), is(POLICY_PAYLOAD));
    assertThat(response.header("flow-variable"), is("value"));
  }

  @Test
  public void varsHaveLocalScopeInSourceAndOperationPolicies() {
    PolicyDefinition policyDefinition =
        policyDefinition(OPERATION_AND_SOURCE_VARS_TEMPLATE_ID, getVariablesApiKey(), propagateMessageTransformations);
    policyDefinition.getConfigurationData().getConfiguration().put("policyId", "p1");
    policyDefinition.getConfigurationData().getConfiguration().put("otherId", "p2");
    policyDefinition.getConfigurationData().getConfiguration().put("payload", POLICY_PAYLOAD);

    PolicyDefinition policyDefinition2 =
        secondPolicy(OPERATION_AND_SOURCE_VARS_TEMPLATE_ID, getVariablesApiKey(), propagateMessageTransformations);
    policyDefinition2.getConfigurationData().getConfiguration().put("policyId", "p2");
    policyDefinition2.getConfigurationData().getConfiguration().put("otherId", "p1");
    policyDefinition2.getConfigurationData().getConfiguration().put("payload", POLICY_PAYLOAD_2);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = request.get();

    assertThat(response.asString(), response.statusCode(), is(200));
    assertThat(response.asString(), is(POLICY_PAYLOAD));
    assertThat(response.header("flow-variable"), is("value"));
  }

  private String backendReceivedBody() {
    return new String(backend.getHttpServer().getLastHttpRequest().getBody());
  }
}
