/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.scope.operation;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.anypoint.test.policy.scope.PolicyScopeTestCase;
import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.http.SimpleHttpServerResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class PolicyOperationScopeMessageAttributesTestCase extends PolicyScopeTestCase {

  private static final String BACKEND_RESPONSE = "ServerPayload";

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(app1, app2, app3)
          .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY,
                                                       "policies/policy-operation-scope-add-request-attributes-policy.xml"),
                               buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_2,
                                                       "policies/policy-operation-scope-add-response-attributes-policy.xml"))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(portApp1)
      .around(portApp2)
      .around(portApp3)
      .around(backend)
      .around(installation);

  private HttpRequest appendPayloadRequest;
  private HttpRequest emptyRequest;
  private HttpRequest requesterWithUrl;

  private FakeGatewayServer server = installation.getServer();

  public PolicyOperationScopeMessageAttributesTestCase(DynamicPort port, ApplicationJar application) {
    this.apiFinder = new ApiFinder(application.getAppConfig());
    this.appendPayloadRequest = request(port, "/api/append-payload");
    this.emptyRequest = request(port, "/api/empty");
    this.requesterWithUrl = request(port, "/api/requester-with-url");
  }

  @Parameterized.Parameters(name = "Propagating message modification: {0}")
  public static Object[][] parameters() {
    return new Object[][] {
        {portApp1, app1},
        {portApp2, app2},
        {portApp3, app3},
    };
  }

  @Before
  public void setUp() {
    backend.getHttpServer().setResponse(SimpleHttpServerResponse.builder().body(BACKEND_RESPONSE).build());
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void attributesPropagateToRequest() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), true);
    configBeforeRequestPolicy(policyDefinition, "policyHeader", "policyHeaderValue", "policyQueryParam",
                              "policyQueryParamValue", "policyUriParam", "/policyUriParamValue", "{policyUriParam}");

    server.deployPolicy(policyDefinition);

    appendPayloadRequest.get();

    assertThat(backendHeader("Flowheader"), is("flowHeaderValue"));
    assertThat(backendHeader("Policyheader"), is("policyHeaderValue"));
    assertThat(backendQueryParam("flowQueryParam"), is("flowQueryParamValue"));
    assertThat(backendQueryParam("policyQueryParam"), is("policyQueryParamValue"));
    assertThat(backendPath(), is("/flowUriParamValue/policyUriParamValue"));
  }

  @Test
  public void attributesPropagateToRequestWhenUsingRequesterWithUrl() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getRequesterWithUrlApiKey(), true);
    configBeforeRequestPolicy(policyDefinition, "policyHeader", "policyHeaderValue", "policyQueryParam",
                              "policyQueryParamValue", "policyUriParam", "/policyUriParamValue", "{policyUriParam}");

    server.deployPolicy(policyDefinition);

    requesterWithUrl.get();

    assertThat(backendHeader("Flowheader"), is("flowHeaderValue"));
    assertThat(backendHeader("Policyheader"), is("policyHeaderValue"));
    assertThat(backendQueryParam("flowQueryParam"), is("flowQueryParamValue"));
    assertThat(backendQueryParam("policyQueryParam"), is("policyQueryParamValue"));
    assertThat(backendPath(), is("/flowUriParamValue"));
  }

  @Test
  public void attributesPropagateToRequestWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), false);
    configBeforeRequestPolicy(policyDefinition, "policyHeader", "policyHeaderValue", "policyQueryParam",
                              "policyQueryParamValue", "policyUriParam", "/policyUriParamValue", "{policyUriParam}");

    server.deployPolicy(policyDefinition);

    appendPayloadRequest.get();

    assertThat(backendHeader("Flowheader"), is("flowHeaderValue"));
    assertThat(backendHeader("Policyheader"), is("policyHeaderValue"));
    assertThat(backendQueryParam("flowQueryParam"), is("flowQueryParamValue"));
    assertThat(backendQueryParam("policyQueryParam"), is("policyQueryParamValue"));
    assertThat(backendPath(), is("/flowUriParamValue/policyUriParamValue"));
  }

  @Test
  public void multiplePoliciesAttributesPropagateToRequest() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), true);
    configBeforeRequestPolicy(policyDefinition, "policyHeader1", "policyHeaderValue1", "policyQueryParam1",
                              "policyQueryParamValue1", "policyUriParam1", "/policyUriParamValue1", "{policyUriParam1}");
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), true);
    configBeforeRequestPolicy(policyDefinition2, "policyHeader2", "policyHeaderValue2", "policyQueryParam2",
                              "policyQueryParamValue2", "policyUriParam2", "/policyUriParamValue2", "{policyUriParam2}");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    appendPayloadRequest.get();

    assertThat(backendHeader("Flowheader"), is("flowHeaderValue"));
    assertThat(backendHeader("Policyheader1"), is("policyHeaderValue1"));
    assertThat(backendHeader("Policyheader2"), is("policyHeaderValue2"));
    assertThat(backendQueryParam("flowQueryParam"), is("flowQueryParamValue"));
    assertThat(backendQueryParam("policyQueryParam1"), is("policyQueryParamValue1"));
    assertThat(backendQueryParam("policyQueryParam2"), is("policyQueryParamValue2"));
    assertThat(backendPath(), is("/flowUriParamValue/policyUriParamValue1/policyUriParamValue2"));
  }

  @Test
  public void multiplePoliciesAttributesPropagateToRequestWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), false);
    configBeforeRequestPolicy(policyDefinition, "policyHeader1", "policyHeaderValue1", "policyQueryParam1",
                              "policyQueryParamValue1", "policyUriParam1", "/policyUriParamValue1", "{policyUriParam1}");
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), false);
    configBeforeRequestPolicy(policyDefinition2, "policyHeader2", "policyHeaderValue2", "policyQueryParam2",
                              "policyQueryParamValue2", "policyUriParam2", "/policyUriParamValue2", "{policyUriParam2}");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    appendPayloadRequest.get();

    assertThat(backendHeader("Flowheader"), is("flowHeaderValue"));
    assertThat(backendHeader("Policyheader1"), is("policyHeaderValue1"));
    assertThat(backendHeader("Policyheader2"), is("policyHeaderValue2"));
    assertThat(backendQueryParam("flowQueryParam"), is("flowQueryParamValue"));
    assertThat(backendQueryParam("policyQueryParam1"), is("policyQueryParamValue1"));
    assertThat(backendQueryParam("policyQueryParam2"), is("policyQueryParamValue2"));
    assertThat(backendPath(), is("/flowUriParamValue/policyUriParamValue1/policyUriParamValue2"));
  }

  @Test
  public void attributesPropagateFromRequesterResponse() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put("headerName", "policyHeader");
    policyDefinition.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue");

    server.deployPolicy(policyDefinition);

    HttpResponse response = emptyRequest.get();

    assertThat(response.statusCode(), is(201));
    assertThat(response.header("policyHeader"), is("policyHeaderValue"));
  }

  @Test
  public void attributesDontPropagateFromRequesterResponseWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put("headerName", "policyHeader");
    policyDefinition.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue");

    server.deployPolicy(policyDefinition);

    HttpResponse response = emptyRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header("policyHeader"), nullValue());
  }

  @Test
  public void multiplePoliciesAttributesPropagateFromRequesterResponse() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put("headerName", "policyHeader1");
    policyDefinition.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue1");
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), true);
    policyDefinition2.getConfigurationData().getConfiguration().put("headerName", "policyHeader2");
    policyDefinition2.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue2");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = emptyRequest.get();

    assertThat(response.statusCode(), is(201));
    assertThat(response.header("policyHeader1"), is("policyHeaderValue1"));
    assertThat(response.header("policyHeader2"), is("policyHeaderValue2"));
  }

  @Test
  public void multiplePoliciesAttributesDontPropagateFromRequesterResponseWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put("headerName", "policyHeader1");
    policyDefinition.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue1");
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), false);
    policyDefinition2.getConfigurationData().getConfiguration().put("headerName", "policyHeader2");
    policyDefinition2.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue2");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = emptyRequest.get();

    assertThat(response.header("policyHeader1"), nullValue());
    assertThat(response.header("policyHeader2"), nullValue());
    assertThat(response.statusCode(), is(200));
  }

  @Test
  public void multiplePoliciesAttributesMixedPropagateFromRequesterResponse() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put("headerName", "policyHeader1");
    policyDefinition.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue1");
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), false);
    policyDefinition2.getConfigurationData().getConfiguration().put("headerName", "policyHeader2");
    policyDefinition2.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue2");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = emptyRequest.get();

    assertThat(response.statusCode(), is(201));
    assertThat(response.header("policyHeader1"), is("policyHeaderValue1"));
    assertThat(response.header("policyHeader2"), nullValue());
  }

  @Test
  public void multiplePoliciesAttributesMixedPropagateFromRequesterResponseInverted() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put("headerName", "policyHeader1");
    policyDefinition.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue1");
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getEmptyApiKey(), true);
    policyDefinition2.getConfigurationData().getConfiguration().put("headerName", "policyHeader2");
    policyDefinition2.getConfigurationData().getConfiguration().put("headerValue", "policyHeaderValue2");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = emptyRequest.get();

    assertThat(response.statusCode(), is(201));
    assertThat(response.header("policyHeader1"), nullValue());
    assertThat(response.header("policyHeader2"), is("policyHeaderValue2"));
  }

  private void configBeforeRequestPolicy(PolicyDefinition policyDefinition, String headerName, String headerValue,
                                         String queryParamName, String queryParamValue, String uriParamName, String uriParamValue,
                                         String requestPath) {
    policyDefinition.getConfigurationData().getConfiguration().put("headerName", headerName);
    policyDefinition.getConfigurationData().getConfiguration().put("headerValue", headerValue);
    policyDefinition.getConfigurationData().getConfiguration().put("queryParamName", queryParamName);
    policyDefinition.getConfigurationData().getConfiguration().put("queryParamValue", queryParamValue);
    policyDefinition.getConfigurationData().getConfiguration().put("uriParamName", uriParamName);
    policyDefinition.getConfigurationData().getConfiguration().put("uriParamValue", uriParamValue);
    policyDefinition.getConfigurationData().getConfiguration().put("requestPath", requestPath);
  }

  private String backendHeader(String headerName) {
    return backend.getHttpServer().getLastHttpRequest().getHeaders().get(headerName).stream().findFirst().orElse(null);
  }

  private String backendQueryParam(String queryParamName) {
    return backend.getHttpServer().getLastHttpRequest().getQueryParams().get(queryParamName);
  }

  private String backendPath() {
    return backend.getHttpServer().getLastHttpRequest().getUri();
  }
}
