/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.scope.operation;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.GROUP_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.anypoint.test.policy.scope.PolicyScopeTestCase;
import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.http.SimpleHttpServerResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class PolicyOperationScopePayloadTestCase extends PolicyScopeTestCase {

  private static final PolicyTemplateKey PAYLOAD_UNTOUCHED_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "payloadUntouched", "0.1.0");
  private static final PolicyTemplateKey ALL_IN_ONE_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "allInOne", "0.1.0");

  private static final String PAYLOAD_PARAM = "policyPayload";
  private static final String PAYLOAD = "-policyPayloadValue-";
  private static final String ANOTHER_PAYLOAD = "-anotherPolicyPayloadValue-";
  private static final String BEFORE_FLOW_REQUEST_PAYLOAD = "-Before Request Flow Payload-";
  private static final String AFTER_FLOW_PAYLOAD = "-After Request Flow Payload-";

  private static final String BACKEND_RESPONSE = "ServerPayload";

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(app1, app2, app3)
          .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY,
                                                       "policies/policy-operation-scope-payload-before-policy.xml"),
                               buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_2,
                                                       "policies/policy-operation-scope-payload-after-policy.xml"),
                               buildTestPolicyTemplate(PAYLOAD_UNTOUCHED_TEMPLATE_ID,
                                                       "policies/policy-operation-scope-payload-untouched-policy.xml"),
                               buildTestPolicyTemplate(ALL_IN_ONE_TEMPLATE_ID,
                                                       "policies/policy-operation-scope-all-in-one-payload-policy.xml"))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(portApp1)
      .around(portApp2)
      .around(portApp3)
      .around(backend)
      .around(installation);

  private HttpRequest appendPayloadRequest;
  private HttpRequest emptyRequest;

  private FakeGatewayServer server = installation.getServer();

  public PolicyOperationScopePayloadTestCase(DynamicPort port, ApplicationJar application) {
    this.apiFinder = new ApiFinder(application.getAppConfig());
    this.appendPayloadRequest = request(port, "/api/append-payload");
    this.emptyRequest = request(port, "/api/empty");
  }

  @Parameterized.Parameters(name = "Propagating message modification: {0}")
  public static Object[][] parameters() {
    return new Object[][] {
        {portApp1, app1},
        {portApp2, app2},
        {portApp3, app3},
    };
  }

  @Before
  public void setUp() {
    backend.getHttpServer().setResponse(SimpleHttpServerResponse.builder().body(BACKEND_RESPONSE).build());
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void payloadPropagatesToRequest() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);

    server.deployPolicy(policyDefinition);

    appendPayloadRequest.get();

    assertThat(backendResponse(), is(BEFORE_FLOW_REQUEST_PAYLOAD + PAYLOAD));
  }

  @Test
  public void payloadPropagatesToRequestWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);

    server.deployPolicy(policyDefinition);

    appendPayloadRequest.get();

    assertThat(backendResponse(), is(BEFORE_FLOW_REQUEST_PAYLOAD + PAYLOAD));
  }

  @Test
  public void multiplePoliciesPayloadPropagatesToRequest() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), true);
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);
    policyDefinition2.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, ANOTHER_PAYLOAD);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    appendPayloadRequest.get();

    assertThat(backendResponse(), is(BEFORE_FLOW_REQUEST_PAYLOAD + PAYLOAD + ANOTHER_PAYLOAD));
  }

  @Test
  public void multiplePoliciesPayloadPropagatesToRequestWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), false);
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY, getAppendPayloadApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);
    policyDefinition2.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, ANOTHER_PAYLOAD);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    appendPayloadRequest.get();

    assertThat(backendResponse(), is(BEFORE_FLOW_REQUEST_PAYLOAD + PAYLOAD + ANOTHER_PAYLOAD));
  }

  @Test
  public void payloadPropagatesFromRequesterResponse() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);

    server.deployPolicy(policyDefinition);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(response.asString(), is(BACKEND_RESPONSE + PAYLOAD + AFTER_FLOW_PAYLOAD));
  }

  @Test
  public void payloadDoesNotPropagateFromRequesterResponseWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);

    server.deployPolicy(policyDefinition);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(response.asString(), is(BACKEND_RESPONSE + AFTER_FLOW_PAYLOAD));
  }

  @Test
  public void multiplePoliciesPayloadPropagatesFromRequesterResponse() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), true);
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);
    policyDefinition2.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, ANOTHER_PAYLOAD);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(response.asString(), is(BACKEND_RESPONSE + ANOTHER_PAYLOAD + PAYLOAD + AFTER_FLOW_PAYLOAD));
  }

  @Test
  public void multiplePoliciesPayloadDoesNotPropagateFromRequesterResponseWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), false);
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);
    policyDefinition2.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, ANOTHER_PAYLOAD);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(response.asString(), is(BACKEND_RESPONSE + AFTER_FLOW_PAYLOAD));
  }

  @Test
  public void multiplePoliciesPayloadMixedPropagationFromRequesterResponse() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), false);
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);
    policyDefinition2.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, ANOTHER_PAYLOAD);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(response.asString(), is(BACKEND_RESPONSE + ANOTHER_PAYLOAD + AFTER_FLOW_PAYLOAD));
  }

  @Test
  public void multiplePoliciesPayloadMixedPropagationFromRequesterResponseInverted() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), true);
    PolicyDefinition policyDefinition2 = secondPolicy(POLICY_TEMPLATE_KEY_2, getAppendPayloadApiKey(), false);
    policyDefinition.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, PAYLOAD);
    policyDefinition2.getConfigurationData().getConfiguration().put(PAYLOAD_PARAM, ANOTHER_PAYLOAD);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(response.asString(), is(BACKEND_RESPONSE + PAYLOAD + AFTER_FLOW_PAYLOAD));
  }

  @Test
  public void allInOne() {
    PolicyDefinition policyDefinition = policyDefinition(ALL_IN_ONE_TEMPLATE_ID, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put("sourcePropagationEnabled", "true");
    policyDefinition.getConfigurationData().getConfiguration().put("operationPropagationEnabled", "true");

    server.deployPolicy(policyDefinition);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(backendResponse(), is("1" + BEFORE_FLOW_REQUEST_PAYLOAD + "-3"));
    assertThat(response.asString(), is(BACKEND_RESPONSE + "-4" + AFTER_FLOW_PAYLOAD + "-2"));
  }

  @Test
  public void allInOneSourcePropagatesOperationDont() {
    PolicyDefinition policyDefinition = policyDefinition(ALL_IN_ONE_TEMPLATE_ID, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put("sourcePropagationEnabled", "true");
    policyDefinition.getConfigurationData().getConfiguration().put("operationPropagationEnabled", "false");

    server.deployPolicy(policyDefinition);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(backendResponse(), is("1" + BEFORE_FLOW_REQUEST_PAYLOAD + "-3"));
    assertThat(response.asString(), is(BACKEND_RESPONSE + AFTER_FLOW_PAYLOAD + "-2"));
  }

  @Test
  public void allInOneOperationPropagatesSourceDont() {
    PolicyDefinition policyDefinition = policyDefinition(ALL_IN_ONE_TEMPLATE_ID, getAppendPayloadApiKey(), true);
    policyDefinition.getConfigurationData().getConfiguration().put("sourcePropagationEnabled", "false");
    policyDefinition.getConfigurationData().getConfiguration().put("operationPropagationEnabled", "true");

    server.deployPolicy(policyDefinition);

    HttpResponse response = appendPayloadRequest.get();

    assertThat(backendResponse(), is(BEFORE_FLOW_REQUEST_PAYLOAD + "-3"));
    assertThat(response.asString(), is(BACKEND_RESPONSE + "-4" + AFTER_FLOW_PAYLOAD + "-2"));
  }

  @Test
  public void payloadUntouchedWhenPolicyDoesNotChangeIt() {
    PolicyDefinition policyDefinition = policyDefinition(PAYLOAD_UNTOUCHED_TEMPLATE_ID, getEmptyApiKey(), false);

    server.deployPolicy(policyDefinition);

    assertThat(emptyRequest.withPayload("Payload").post().asString(), is(BACKEND_RESPONSE));
    assertThat(backendResponse(), is("Payload"));
  }

  private String backendResponse() {
    return new String(backend.getHttpServer().getLastHttpRequest().getBody());
  }
}
