/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.scope.source;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.GROUP_ID;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.isEmptyString;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;

import com.mulesoft.anypoint.test.policy.scope.PolicyScopeTestCase;
import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class PolicyScopeMessageAttributesTestCase extends PolicyScopeTestCase {

  private static final ApiKey EMPTY_API_KEY = new ApiKey(1L);
  private static final ApiKey READ_ATTR_API_KEY = new ApiKey(6L);
  private static final ApiKey SET_ATTR_API_KEY = new ApiKey(7L);

  private static final PolicyTemplateKey ADD_REQUEST_HEADER_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "addRequestHeader", "0.1.0");
  private static final PolicyTemplateKey READ_REQUEST_HEADER_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "readRequestHeader", "0.1.0");
  private static final PolicyTemplateKey ADD_RESPONSE_HEADER_TEMPLATE_ID =
      new PolicyTemplateKey(GROUP_ID, "addResponseHeader", "0.1.0");

  private static final String HEADER_NAME_PARAM = "headerName";
  private static final String HEADER_VALUE_PARAM = "headerValue";
  private static final String HEADER_NAME = "headerName";
  private static final String HEADER_NAME_2 = "headerName2";
  private static final String FLOW_HEADER_NAME = "flowHeaderName";
  private static final String FLOW_HEADER_VALUE = "flowHeaderValue";
  private static final String POLICY_HEADER_VALUE = "policyHeaderValue";
  private static final String POLICY_HEADER_VALUE_2 = "policyHeaderValue2";

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(buildTestApplication(APP, "mule-config-http.xml"))
          .withPolicyTemplates(buildTestPolicyTemplate(ADD_REQUEST_HEADER_TEMPLATE_ID,
                                                       "policies/policy-scope-add-request-attributes-policy.xml"),
                               buildTestPolicyTemplate(READ_REQUEST_HEADER_TEMPLATE_ID,
                                                       "policies/policy-scope-read-request-attributes-policy.xml"),
                               buildTestPolicyTemplate(ADD_RESPONSE_HEADER_TEMPLATE_ID,
                                                       "policies/policy-scope-add-response-attributes-policy.xml"))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(portApp1)
      .around(installation);

  private HttpRequest readAttributesRequest = request(portApp1, "/api/read-attributes");
  private HttpRequest setAttributesRequest = request(portApp1, "/api/set-attributes");
  private HttpRequest emptyRequest = request(portApp1, "/api/empty");

  private FakeGatewayServer server = installation.getServer();

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void addedRequestHeaderPropagatesToFlow() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, true);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE);

    server.deployPolicy(policyDefinition);

    HttpResponse response = readAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), is(POLICY_HEADER_VALUE));
  }

  @Test
  public void addedRequestHeaderDoesNotPropagateToFlowWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, false);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE);

    server.deployPolicy(policyDefinition);

    HttpResponse response = readAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), nullValue());
  }

  @Test
  public void addedRequestHeaderPropagatesToNextPolicy() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_REQUEST_HEADER_TEMPLATE_ID, EMPTY_API_KEY, true);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE);
    PolicyDefinition policyDefinition2 = secondPolicy(READ_REQUEST_HEADER_TEMPLATE_ID, EMPTY_API_KEY, true);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    assertThat(emptyRequest.get().asString(), is(POLICY_HEADER_VALUE));
  }

  @Test
  public void addedRequestHeaderDoesNotPropagateToNextPolicyWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, false);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE);
    PolicyDefinition policyDefinition2 = secondPolicy(READ_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, true);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = readAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), nullValue());
    assertThat(readAttributesRequest.get().asString(), isEmptyString());
  }

  @Test
  public void onlyHeaderOfPropagationEnabledPolicyReachesTheFlow() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, false);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE);
    PolicyDefinition policyDefinition2 = secondPolicy(ADD_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, true);
    policyDefinition2.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME_2);
    policyDefinition2.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE_2);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = readAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), nullValue());
    assertThat(response.header(HEADER_NAME_2), is(POLICY_HEADER_VALUE_2));
  }

  @Test
  public void onlyHeaderOfPropagationEnabledPolicyReachesTheFlowInverted() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, true);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE);
    PolicyDefinition policyDefinition2 = secondPolicy(ADD_REQUEST_HEADER_TEMPLATE_ID, READ_ATTR_API_KEY, false);
    policyDefinition2.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME_2);
    policyDefinition2.getConfigurationData().getConfiguration().put(HEADER_VALUE_PARAM, POLICY_HEADER_VALUE_2);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = readAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), is(POLICY_HEADER_VALUE));
    assertThat(response.header(HEADER_NAME_2), nullValue());
  }

  @Test
  public void addResponseHeadersAreAddedToExistingOnes() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_RESPONSE_HEADER_TEMPLATE_ID, SET_ATTR_API_KEY, true);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);

    server.deployPolicy(policyDefinition);

    HttpResponse response = setAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), is(POLICY_HEADER_VALUE));
    assertThat(response.header(FLOW_HEADER_NAME), is(FLOW_HEADER_VALUE));
  }

  @Test
  public void addResponseHeadersAreAddedToExistingOnesEvenWhenDisabled() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_RESPONSE_HEADER_TEMPLATE_ID, SET_ATTR_API_KEY, false);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);

    server.deployPolicy(policyDefinition);

    HttpResponse response = setAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), is(POLICY_HEADER_VALUE));
    assertThat(response.header(FLOW_HEADER_NAME), is(FLOW_HEADER_VALUE));
  }

  @Test
  public void addResponseHeadersMultiplePoliciesAreAddedToExistingOnes() {
    PolicyDefinition policyDefinition = policyDefinition(ADD_RESPONSE_HEADER_TEMPLATE_ID, SET_ATTR_API_KEY, false);
    policyDefinition.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME);
    PolicyDefinition policyDefinition2 = secondPolicy(ADD_RESPONSE_HEADER_TEMPLATE_ID, SET_ATTR_API_KEY, false);
    policyDefinition2.getConfigurationData().getConfiguration().put(HEADER_NAME_PARAM, HEADER_NAME_2);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    HttpResponse response = setAttributesRequest.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(HEADER_NAME), is(POLICY_HEADER_VALUE));
    assertThat(response.header(HEADER_NAME_2), is(POLICY_HEADER_VALUE));
    assertThat(response.header(FLOW_HEADER_NAME), is(FLOW_HEADER_VALUE));
  }

}
