/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics;

import static com.mulesoft.mule.runtime.gw.api.analytics.GatewayAnalytics.GW_ANALYTICS_REGISTRY_KEY;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.config.custom.CustomizationService;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.notification.ConnectorMessageNotification;
import org.mule.runtime.api.notification.ConnectorMessageNotificationListener;
import org.mule.runtime.api.notification.NotificationListenerRegistry;
import org.mule.runtime.container.api.MuleCoreExtensionDependency;
import org.mule.runtime.core.api.context.notification.ServerNotificationManager;
import org.mule.runtime.module.deployment.api.DeploymentService;

import com.mulesoft.mule.runtime.gw.analytics.api.DefaultGatewayAnalytics;
import com.mulesoft.mule.runtime.gw.analytics.cache.AnalyticsEventCacheManager;
import com.mulesoft.mule.runtime.gw.analytics.cache.AnalyticsEventCaches;
import com.mulesoft.mule.runtime.gw.analytics.notification.AnalyticsConnectorMessageNotificationListener;
import com.mulesoft.mule.runtime.gw.analytics.notification.AnalyticsNotificationListenerSupplier;
import com.mulesoft.mule.runtime.gw.api.service.EventsCollectorService;
import com.mulesoft.mule.runtime.gw.autodiscovery.ApiDiscovery;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.factory.VariableExecutorBackoffSchedulerFactory;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.deployment.ApiDeploymentCoreExtension;
import com.mulesoft.mule.runtime.gw.extension.GatewayEntitledCoreExtension;
import com.mulesoft.mule.runtime.gw.logging.LoggingClassLoaderSelector;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AnalyticsCoreExtension extends GatewayEntitledCoreExtension {

  // These values can not be changed, as they are used by Mule Agent
  public static final String ANALYTICS_QUEUE = "_analyticsQueue";
  public static final String ANALYTICS_POLICY_VIOLATION_QUEUE = "_analyticsPolicyViolationQueue";

  private static final Logger LOGGER = LoggerFactory.getLogger(AnalyticsCoreExtension.class);

  @Inject
  private DeploymentService deploymentService;

  @Inject
  private EventsCollectorService eventsCollectorService;

  private ApiDeploymentCoreExtension apiDeploymentCoreExtension;

  private AnalyticsPollersManager pollersManager;

  private AnalyticsEventHandlerManager eventHandlerManager;
  private AnalyticsEventCacheManager cacheManager;
  private AnalyticsConfiguration configuration;
  private ApiDiscovery apiDiscovery;

  private boolean extensionInitialized;

  public AnalyticsCoreExtension() {
    this.configuration = new AnalyticsConfiguration();
    this.apiDiscovery = new ApiDiscovery();
  }

  @Override
  public String getName() {
    return "Analytics Collector Extension";
  }

  @Override
  public void initialiseCoreExtension() {}

  @Override
  public void start() {}

  @Override
  public void onArtifactCreated(String artifactName, CustomizationService customizationService) {
    LoggingClassLoaderSelector.initialise(Thread.currentThread().getContextClassLoader());
    withContextClassLoader(containerClassLoader.getClassLoader(), () -> {
      if (analyticsEnabled() && extensionLoaded() && onlineMode()) {

        customizationService.registerCustomServiceImpl("analytics-queues-registration",
                                                       new ExtensionInitialisation(artifactName, this));
        customizationService.registerCustomServiceImpl(GW_ANALYTICS_REGISTRY_KEY,
                                                       new DefaultGatewayAnalytics(apiDeploymentCoreExtension.getApiService()));

        initializeCache();

        cacheManager.getAgentCaches().ifPresent(caches -> {
          customizationService.registerCustomServiceImpl(ANALYTICS_QUEUE, caches.getRegularEventsCache());
          customizationService.registerCustomServiceImpl(ANALYTICS_POLICY_VIOLATION_QUEUE, caches.getViolationsEventsCache());
        });
      }
    });
  }

  @Override
  public void onUndeploymentStart(String artifactName) {
    if (eventHandlerManager != null) {
      eventHandlerManager.remove(artifactName);
    }
  }

  @Override
  public void stop() {
    if (pollersManager != null) {
      pollersManager.shutdown();
    }
  }

  @Override
  public void dispose() {
    if (cacheManager != null) {
      cacheManager.dispose();
    }
  }

  @MuleCoreExtensionDependency
  public void setApiDeploymentCoreExtension(ApiDeploymentCoreExtension apiDeploymentCoreExtension) {
    this.apiDeploymentCoreExtension = apiDeploymentCoreExtension;
  }

  /**
   * Synchronized method to avoid multiple initializations when parallel deployment is enabled
   */
  private synchronized void finishLazyInitialization() {
    if (!extensionInitialized) {
      LOGGER.debug("Starting {}", getName());

      eventHandlerManager =
          new AnalyticsEventHandlerManager(apiDeploymentCoreExtension.getApiService(), cacheManager, eventsCollectorService,
                                           configuration.isServiceMesh(), apiDeploymentCoreExtension.metricsCollector());

      deploymentService.addDomainDeploymentListener(this);

      cacheManager.getCloudCaches().ifPresent(this::startSchedulers);

      apiDeploymentCoreExtension.getApiService().addDeploymentListener(new AnalyticsApiDeploymentListener());
      apiDeploymentCoreExtension.getNotificationListenerSuppliers()
          .add(new AnalyticsNotificationListenerSupplier(eventsCollectorService));

      extensionInitialized = true;
    }
  }

  /**
   * Synchronized method to avoid multiple initializations when parallel deployment is enabled
   */
  private synchronized void initializeCache() {
    if (cacheManager == null) {
      cacheManager = new AnalyticsEventCacheManager(configuration);
      cacheManager.initialise();
    }
  }

  private void startSchedulers(AnalyticsEventCaches caches) {
    pollersManager = new AnalyticsPollersManager(configuration, caches, apiDeploymentCoreExtension.apiPlatformClient(),
                                                 new VariableExecutorBackoffSchedulerFactory());
    pollersManager.scheduleAnalyticsRunnable();
    pollersManager.schedulePolicyViolationsRunnable();
  }

  private boolean analyticsEnabled() {
    return configuration.isEnabled();
  }

  /**
   * Custom service used to be able to recover {@link com.mulesoft.mule.runtime.gw.api.key.ApiKey}s from the application's
   * {@link Registry}, so the core extension initialisation can be done only after an application contains an API in it.
   */
  class ExtensionInitialisation implements Initialisable {

    private final String artifactName;
    private final AnalyticsCoreExtension analyticsCoreExtension;

    @Inject
    private ServerNotificationManager notificationManager;

    @Inject
    private Registry registry;

    public ExtensionInitialisation(String artifactName, AnalyticsCoreExtension analyticsCoreExtension) {
      this.artifactName = artifactName;
      this.analyticsCoreExtension = analyticsCoreExtension;
    }

    @Override
    public void initialise() {
      withContextClassLoader(containerClassLoader.getClassLoader(), () -> {
        if (!apiDiscovery.apiKeys(registry).isEmpty()) {

          analyticsCoreExtension.finishLazyInitialization();

          AnalyticsEventHandler eventHandler = eventHandlerManager.create(artifactName);

          registry.lookupByType(NotificationListenerRegistry.class).get()
              .registerListener(new AnalyticsConnectorMessageNotificationListener(eventHandler));

          if (!notificationManager.getInterfaceToTypes().containsKey(ConnectorMessageNotificationListener.class)) {
            notificationManager.addInterfaceToType(ConnectorMessageNotificationListener.class,
                                                   ConnectorMessageNotification.class);
          }
        }
      });
    }

  }
}
