/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.mule.runtime.gw.api.analytics.GatewayAnalytics.GW_ANALYTICS_REGISTRY_KEY;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Optional.of;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.config.custom.CustomizationService;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.notification.NotificationListenerRegistry;
import org.mule.runtime.core.api.context.notification.ServerNotificationManager;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.mule.runtime.gw.analytics.AnalyticsCoreExtension.ExtensionInitialisation;
import com.mulesoft.mule.runtime.gw.autodiscovery.ByIdAutodiscovery;
import com.mulesoft.mule.runtime.gw.autodiscovery.api.Autodiscovery;
import com.mulesoft.mule.runtime.gw.deployment.ApiDeploymentCoreExtension;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import java.util.Arrays;
import java.util.Collection;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.mockito.ArgumentCaptor;

@RunWith(Parameterized.class)
public class AnalyticsCoreExtensionTestCase extends AbstractMuleTestCase {

  private static final String CLOUD = "Cloud";
  private static final String AGENT = "Agent";
  private static final String CLOUD_AND_AGENT = "Cloud & Agent";
  private static final String MESH = "Mesh";

  @Rule
  public SystemPropertyTemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  private boolean cloudEnabled;
  private boolean multipleConsumers;
  private boolean meshEnabled;
  private Asserter asserter;

  private CustomizationService customizationService;
  private ApiDeploymentCoreExtension apiDeploymentCoreExtension;

  private AnalyticsCoreExtension analyticsCoreExtension;
  private ArgumentCaptor<ExtensionInitialisation> argumentCaptor;

  public AnalyticsCoreExtensionTestCase(String mode) {
    if (CLOUD.equals(mode)) {
      this.cloudEnabled = true;
      this.multipleConsumers = false;
      this.meshEnabled = false;
      this.asserter = cloudStarted();
    } else if (AGENT.equals(mode)) {
      this.cloudEnabled = false;
      this.multipleConsumers = false;
      this.meshEnabled = false;
      this.asserter = agentStarted();
    } else if (CLOUD_AND_AGENT.equals(mode)) {
      this.cloudEnabled = true;
      this.multipleConsumers = true;
      this.meshEnabled = false;
      this.asserter = cloudAndAgentStarted();
    } else if (MESH.equals(mode)) {
      this.cloudEnabled = true;
      this.multipleConsumers = false;
      this.meshEnabled = true;
      this.asserter = meshStarted();
    } else {
      throw new IllegalStateException("Invalid mode: " + mode);
    }
  }

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {CLOUD},
        {AGENT},
        {CLOUD_AND_AGENT},
        {MESH}
    });
  }

  @Before
  public void setUp() {
    customizationService = mock(CustomizationService.class);
    argumentCaptor = ArgumentCaptor.forClass(ExtensionInitialisation.class);

    ArtifactClassLoader artifactClassLoader = mock(ArtifactClassLoader.class);

    when(artifactClassLoader.getClassLoader()).thenReturn(Thread.currentThread().getContextClassLoader());

    apiDeploymentCoreExtension = mock(ApiDeploymentCoreExtension.class, RETURNS_DEEP_STUBS);

    analyticsCoreExtension = new AnalyticsCoreExtension();
    analyticsCoreExtension.setApiDeploymentCoreExtension(apiDeploymentCoreExtension);
    analyticsCoreExtension.setContainerClassLoader(artifactClassLoader);
    injectDeploymentService(mock(DeploymentService.class));

    System.setProperty("anypoint.platform.analytics_base_uri", cloudEnabled ? "analyticsUri" : "");
    System.setProperty("anypoint.platform.analytics_multiple_consumers_enabled", multipleConsumers ? "true" : "false");
    System.setProperty("anypoint.platform.analytics_mode", meshEnabled ? "service_mesh" : "classic");
  }

  @After
  public void tearDown() {
    System.clearProperty("anypoint.platform.analytics_enabled");
    System.clearProperty("anypoint.platform.analytics_base_uri");
    System.clearProperty("anypoint.platform.analytics_multiple_consumers_enabled");
    System.clearProperty("anypoint.platform.analytics_mode");
    System.clearProperty("anypoint.platform.client_id");
    System.clearProperty("anypoint.platform.client_secret");
  }

  @Test
  public void unDeployingNotInitializedCausesNoException() throws InitialisationException {
    analyticsCoreExtension.initialise();
    analyticsCoreExtension.onUndeploymentStart("dummy-artifact");
  }

  @Test
  public void analyticsStartedWhenRuntimeInOnlineMode() throws Exception {
    onlineRuntime();

    initialiseCoreExtension();
    appDeployedWithApi();

    asserter.analyticsStarted();
  }

  @Test
  public void analyticsNotStartedBeforeAppDeployment() throws MuleException {
    onlineRuntime();

    initialiseCoreExtension();

    analyticsNotStarted();
  }

  @Test
  public void analyticsNotStartedWhenAppDoesNotHaveApis() throws MuleException {
    onlineRuntime();

    initialiseCoreExtension();
    appDeployedWithoutApis();

    assertThat(analyticsSchedulerStarted(), is(false));
  }

  @Test
  public void analyticsDisabledWhenRuntimeInOfflineMode() throws Exception {
    initialiseCoreExtension();
    analyticsCoreExtension.onArtifactCreated(APP, customizationService);

    verify(customizationService, never()).registerCustomServiceImpl(eq("analytics-queues-registration"), any());
    analyticsNotStarted();
  }

  @Test
  public void analyticsDisabled() throws Exception {
    System.setProperty("anypoint.platform.analytics_enabled", "false");

    initialiseCoreExtension();
    analyticsCoreExtension.onArtifactCreated(APP, customizationService);

    verify(customizationService, never()).registerCustomServiceImpl(eq("analytics-queues-registration"), any());
    analyticsNotStarted();
  }

  @Test
  public void analyticsCoreExtensionInitialisedOnlyOnce() throws Exception {
    onlineRuntime();

    initialiseCoreExtension();
    appDeployedWithApi();

    verify(apiDeploymentCoreExtension.getNotificationListenerSuppliers(), times(1)).add(any());

    anotherAppDeployedWithApi();

    verify(apiDeploymentCoreExtension.getNotificationListenerSuppliers(), times(1)).add(any());
  }

  private void onlineRuntime() {
    System.setProperty("anypoint.platform.client_id", "clientId");
    System.setProperty("anypoint.platform.client_secret", "clientSecret");
  }

  private void initialiseCoreExtension() throws MuleException {
    analyticsCoreExtension.initialise();
    analyticsCoreExtension.start();
  }

  private void appDeployedWithApi() {
    ByIdAutodiscovery api = new ByIdAutodiscovery();
    api.setId(1L);
    appDeployed(APP, customizationService, api);
  }

  private void appDeployedWithoutApis() {
    appDeployed(APP, customizationService);
  }

  private void anotherAppDeployedWithApi() {
    ByIdAutodiscovery api = new ByIdAutodiscovery();
    api.setId(1L);
    appDeployed(APP_2, mock(CustomizationService.class));
  }

  private void appDeployed(String appName, CustomizationService customizationService, ByIdAutodiscovery... apis) {
    analyticsCoreExtension.onArtifactCreated(appName, customizationService);
    verify(customizationService).registerCustomServiceImpl(eq("analytics-queues-registration"), argumentCaptor.capture());
    injectDependencies(argumentCaptor.getValue(), registry(apis));
    argumentCaptor.getValue().initialise();
  }

  private void injectDependencies(ExtensionInitialisation extensionInitialisation, Registry registry) {
    ServerNotificationManager notificationManager = mock(ServerNotificationManager.class, RETURNS_DEEP_STUBS);

    overrideVariable("registry").in(extensionInitialisation).with(registry);
    overrideVariable("notificationManager").in(extensionInitialisation).with(notificationManager);
  }

  private Registry registry(ByIdAutodiscovery... apis) {
    Registry registry = mock(Registry.class);
    NotificationListenerRegistry listenerRegistry = mock(NotificationListenerRegistry.class, RETURNS_DEEP_STUBS);

    when(registry.lookupByType(NotificationListenerRegistry.class)).thenReturn(of(listenerRegistry));
    when(registry.lookupAllByType(Autodiscovery.class)).thenReturn(newArrayList(apis));

    return registry;
  }

  private void injectDeploymentService(DeploymentService deploymentService) {
    overrideVariable("deploymentService").in(analyticsCoreExtension).with(deploymentService);
  }

  private void analyticsNotStarted() {
    assertThat(analyticsSchedulerStarted(), is(false));
    verify(apiDeploymentCoreExtension.getNotificationListenerSuppliers(), never()).add(any());
    verify(apiDeploymentCoreExtension.getApiService(), never()).addDeploymentListener(any(AnalyticsApiDeploymentListener.class));
    verify(customizationService, never()).registerCustomServiceImpl(eq("_analyticsQueue"), any());
    verify(customizationService, never()).registerCustomServiceImpl(eq("_analyticsPolicyViolationQueue"), any());
    verify(customizationService, never()).registerCustomServiceImpl(eq(GW_ANALYTICS_REGISTRY_KEY), any());
  }

  private Asserter agentStarted() {
    return () -> {
      assertThat(analyticsSchedulerStarted(), is(false));
      verify(apiDeploymentCoreExtension.getNotificationListenerSuppliers()).add(any());
      verify(apiDeploymentCoreExtension.getApiService()).addDeploymentListener(any(AnalyticsApiDeploymentListener.class));
      verify(customizationService).registerCustomServiceImpl(eq("_analyticsQueue"), any());
      verify(customizationService).registerCustomServiceImpl(eq("_analyticsPolicyViolationQueue"), any());
      verify(customizationService).registerCustomServiceImpl(eq(GW_ANALYTICS_REGISTRY_KEY), any());
    };
  }

  private Asserter cloudStarted() {
    return () -> {
      assertThat(analyticsSchedulerStarted(), is(true));
      verify(apiDeploymentCoreExtension.getNotificationListenerSuppliers()).add(any());
      verify(apiDeploymentCoreExtension.getApiService()).addDeploymentListener(any(AnalyticsApiDeploymentListener.class));
      verify(customizationService, never()).registerCustomServiceImpl(eq("_analyticsQueue"), any());
      verify(customizationService, never()).registerCustomServiceImpl(eq("_analyticsPolicyViolationQueue"), any());
      verify(customizationService).registerCustomServiceImpl(eq(GW_ANALYTICS_REGISTRY_KEY), any());
    };
  }

  private Asserter cloudAndAgentStarted() {
    return () -> {
      assertThat(analyticsSchedulerStarted(), is(true));
      verify(apiDeploymentCoreExtension.getNotificationListenerSuppliers()).add(any());
      verify(apiDeploymentCoreExtension.getApiService()).addDeploymentListener(any(AnalyticsApiDeploymentListener.class));
      verify(customizationService).registerCustomServiceImpl(eq("_analyticsQueue"), any());
      verify(customizationService).registerCustomServiceImpl(eq("_analyticsPolicyViolationQueue"), any());
    };
  }

  private Asserter meshStarted() {
    return () -> {
      assertThat(analyticsSchedulerStarted(), is(true));
      verify(apiDeploymentCoreExtension.getNotificationListenerSuppliers()).add(any());
      verify(apiDeploymentCoreExtension.getApiService()).addDeploymentListener(any(AnalyticsApiDeploymentListener.class));
      verify(customizationService).registerCustomServiceImpl(eq("_analyticsQueue"), any());
      verify(customizationService).registerCustomServiceImpl(eq("_analyticsPolicyViolationQueue"), any());
      verify(customizationService).registerCustomServiceImpl(eq(GW_ANALYTICS_REGISTRY_KEY), any());
    };
  }

  private boolean analyticsSchedulerStarted() {
    return new Inspector(analyticsCoreExtension).readNullSafe("pollersManager.analyticsScheduler") != null;
  }

  private interface Asserter {

    /**
     * Assert that Analytics engine is running. The assertion varies between how Analytics is configured
     */
    void analyticsStarted();
  }

}
