/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.cache;

import static com.google.common.collect.Lists.newArrayList;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueue;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueueFactory;

import java.io.IOException;
import java.util.NoSuchElementException;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.mapdb.DB;
import org.mapdb.DBMaker;

public class RegularEventCacheTestCase extends AbstractMuleTestCase {

  private static final String QUEUE_NAME = "queue";

  @Rule
  public TemporaryFolder temporaryFolder;

  private DB db;

  private AnalyticsEventCache cache;

  @Before
  public void setUp() throws IOException {
    db = makeDB();
    SizeLimitedQueue<AnalyticsHttpEvent> sizeLimitedQueue =
        new SizeLimitedQueueFactory().createCircularQueue(db.getQueue(QUEUE_NAME), 3);
    cache = new AnalyticsEventCache(sizeLimitedQueue, QUEUE_NAME);
  }

  @Test
  public void addReturnsTrueWhenElementAdded() {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);

    boolean added = cache.add(event);

    assertTrue(added);
    assertThat(cache.poll(), is(event));
  }

  @Test
  public void offerReturnsTrueWhenElementAdded() {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);

    boolean added = cache.offer(event);

    assertTrue(added);
    assertThat(cache.poll(), is(event));
  }

  @Test
  public void addMoreThanTotalLimitReplacesOldestOne() {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);
    AnalyticsHttpEvent event2 = mock(AnalyticsHttpEvent.class);
    AnalyticsHttpEvent event3 = mock(AnalyticsHttpEvent.class);
    AnalyticsHttpEvent event4 = mock(AnalyticsHttpEvent.class);

    cache.add(event);
    cache.add(event2);
    cache.add(event3);

    boolean added = cache.add(event4);

    assertTrue(added);
    assertThat(cache.poll(), is(event2));
    assertThat(cache.poll(), is(event3));
    assertThat(cache.poll(), is(event4));
    assertThat(cache.poll(), nullValue());
  }

  @Test
  public void pollReturnsTheEvent() {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);
    cache.add(event);

    AnalyticsHttpEvent retrievedEvent = cache.poll();

    assertThat(retrievedEvent, is(event));
  }

  @Test
  public void removeReturnsTheEvent() {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);
    cache.add(event);

    AnalyticsHttpEvent retrievedEvent = cache.remove();

    assertThat(retrievedEvent, is(event));
  }

  @Test
  public void pollingElementFreesSpace() {
    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));

    assertThat(cache.size(), is(3));

    cache.poll();
    cache.poll();
    cache.poll();

    assertThat(cache.size(), is(0));

    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));

    assertThat(cache.size(), is(3));
  }

  @Test
  public void removingElementFreesSpace() {
    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));

    assertThat(cache.size(), is(3));

    cache.remove();
    cache.remove();
    cache.remove();

    assertThat(cache.size(), is(0));

    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));
    cache.add(mock(AnalyticsHttpEvent.class));

    assertThat(cache.size(), is(3));
  }

  @Test
  public void pollEmptyReturnsNull() {
    assertThat(cache.poll(), nullValue());
  }

  @Test(expected = NoSuchElementException.class)
  public void removeEmptyThrowsException() {
    cache.remove();
  }

  @Test
  public void isEmptyTrueWhenEmpty() {
    assertTrue(cache.isEmpty());
  }

  @Test
  public void isEmptyFalseWhenNotEmpty() {
    cache.add(mock(AnalyticsHttpEvent.class));

    assertFalse(cache.isEmpty());
  }

  @Test
  public void addAllAddsAllElements() {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);
    AnalyticsHttpEvent otherEvent = mock(AnalyticsHttpEvent.class);
    cache.addAll(newArrayList(event, otherEvent));

    assertThat(cache, hasSize(2));
    assertThat(cache.poll(), is(event));
    assertThat(cache.poll(), is(otherEvent));
  }

  @Test(expected = IllegalAccessError.class)
  public void exceptionThrownWhenDBWasClosed() {
    db.close();

    cache.add(mock(AnalyticsHttpEvent.class));
  }

  private DB makeDB() {
    return DBMaker.newMemoryDB().make();
  }

}
