/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.extractor;

import static java.lang.String.format;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Answers.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.when;
import static org.mule.runtime.http.api.HttpHeaders.Names.X_FORWARDED_FOR;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.analytics.model.HttpRequestAttributes;

@RunWith(MockitoJUnitRunner.class)
public class DefaultClientIpExtractorTestCase extends AbstractMuleTestCase {

  private static final String REMOTE_IP = "127.0.0.1";
  private static final String REMOTE_ADDRESS = format("bar/%s:5701", REMOTE_IP);
  private static final String REMOTE_IPv6 = "fe80:0:0:0:884:da6f:277f:2cf2%en0";
  private static final String REMOTE_ADDRESS_IPv6 = format("foo/[%s]:5701", REMOTE_IPv6);

  @Mock(answer = RETURNS_DEEP_STUBS)
  private HttpRequestAttributes httpRequestAttributes;

  private ClientIpExtractor clientIpExtractor = new DefaultClientIpExtractor();

  @Test
  public void buildEventWithXForwardedFor() {
    when(httpRequestAttributes.getHeaders().get(X_FORWARDED_FOR.toLowerCase())).thenReturn("127.0.0.2, 128.0.0.1");
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(REMOTE_ADDRESS);

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is("127.0.0.2"));
  }

  @Test
  public void buildEventWithoutXForwardedForNoSlash() {
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(format("%s:5701", REMOTE_IP));

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is(REMOTE_IP));
  }

  @Test
  public void buildEventWithoutXForwardedForMultipleSlashes() {
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(format("bar///%s:5701", REMOTE_IP));

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is(REMOTE_IP));
  }

  @Test
  public void buildEventWithoutXForwardedForIPv6NoSlash() {
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(format("[%s]:5701", REMOTE_IPv6));

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is(REMOTE_IPv6));
  }

  @Test
  public void buildEventWithoutXForwardedForIPv6MultipleSlashes() {
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(format("foo///[%s]:5701", REMOTE_IPv6));

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is(REMOTE_IPv6));
  }

  @Test
  public void buildEventWithoutXForwardedForIPv6andSlash() {
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(REMOTE_ADDRESS_IPv6);

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is(REMOTE_IPv6));
  }

  @Test
  public void buildEventWithXForwardedForIPv6() {
    when(httpRequestAttributes.getHeaders().get(X_FORWARDED_FOR.toLowerCase())).thenReturn("2001:db8::ff00:42:8329, 128.0.0.1");
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(REMOTE_ADDRESS_IPv6);

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is("2001:db8::ff00:42:8329"));
  }

  @Test
  public void buildEventWithXForwardedForWithNonIp() {
    when(httpRequestAttributes.getHeaders().get(X_FORWARDED_FOR.toLowerCase())).thenReturn("www.google.com, 127.0.0.2");
    when(httpRequestAttributes.getRemoteAddress()).thenReturn(REMOTE_ADDRESS);

    String ip = clientIpExtractor.extractClientIp(httpRequestAttributes);

    assertThat(ip, is(REMOTE_IP));
  }
}
