/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.runnable;

import static com.google.common.collect.Lists.newArrayList;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.backoff.configuration.BackoffConfiguration;
import com.mulesoft.mule.runtime.gw.backoff.session.PlatformBackoffBarrier;
import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.analytics.cache.AnalyticsEventCache;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;

import java.io.IOException;

import org.junit.Before;
import org.junit.Test;

public class AnalyticsEventSenderRunnableTestCase extends AbstractMuleTestCase {

  private static final Integer ANALYTICS_BATCH_SIZE = 100;

  private AnalyticsEventCache cache;
  private ApiPlatformClient restClient;

  private AnalyticsEventSenderRunnable instance;

  @Before
  public void setUp() {
    cache = mock(AnalyticsEventCache.class, RETURNS_DEEP_STUBS);
    restClient = mock(ApiPlatformClient.class);

    BackoffConfiguration backoffConfiguration = new BackoffConfiguration.Builder(new GatewayConfiguration()).build();
    instance = new AnalyticsEventSenderRunnable(cache, ANALYTICS_BATCH_SIZE, restClient, backoffConfiguration,
                                                new PlatformBackoffBarrier());
  }

  @Test
  public void restClientNotConnected() {
    when(restClient.isConnected()).thenReturn(false);

    instance.run();

    verify(restClient).isConnected();
    verifyNoMoreInteractions(cache, restClient);
  }

  @Test
  public void emptyCache() throws IOException {
    when(cache.isEmpty()).thenReturn(true);

    instance.run();

    verify(restClient, never()).postHttpEvents(any());
  }

  @Test
  public void sendEvents() throws IOException {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);
    AnalyticsHttpEvent otherEvent = mock(AnalyticsHttpEvent.class);
    when(cache.poll()).thenReturn(event).thenReturn(otherEvent).thenReturn(null);
    when(restClient.isConnected()).thenReturn(true);

    instance.run();

    verify(restClient).postHttpEvents(newArrayList(event, otherEvent));
  }

  @Test
  public void exceptionWhileSendingEventsReAddsThem() throws IOException {
    AnalyticsHttpEvent event = mock(AnalyticsHttpEvent.class);
    AnalyticsHttpEvent otherEvent = mock(AnalyticsHttpEvent.class);
    when(cache.poll()).thenReturn(event).thenReturn(otherEvent).thenReturn(null);
    when(restClient.isConnected()).thenReturn(true);
    doThrow(new RuntimeException()).when(restClient).postHttpEvents(any());

    instance.run();

    verify(restClient).postHttpEvents(newArrayList(event, otherEvent));
    verify(cache).addAll(newArrayList(event, otherEvent));
  }

}
