/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.runnable;

import com.mulesoft.mule.runtime.gw.analytics.cache.AnalyticsEventCache;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.runnable.BackoffRunnable;
import com.mulesoft.anypoint.backoff.session.BackoffBarrier;
import com.mulesoft.anypoint.backoff.session.SessionMetadata;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.exception.NormalizedExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.session.metadata.ApiPlatformSessionMetadata;
import com.mulesoft.anypoint.retry.session.ErrorSessionMetadata;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AnalyticsEventSenderRunnable extends BackoffRunnable {

  private static final String SEND_ANALYTICS_DESCRIPTION = "send events to analytics ingest from";

  private static final Logger LOGGER = LoggerFactory.getLogger(AnalyticsEventSenderRunnable.class);

  private final NormalizedExceptionMessageLogger normalizedLogger = new RecoverableExceptionMessageLogger(LOGGER);
  private final ApiPlatformClient restClient;
  private final int batchSize;

  private AnalyticsEventCache cache;

  public AnalyticsEventSenderRunnable(AnalyticsEventCache eventRepository, int batchSize, ApiPlatformClient restClient,
                                      BackoffConfiguration backoffConfiguration, BackoffBarrier backoffBarrier) {
    super(backoffConfiguration, backoffBarrier);
    this.cache = eventRepository;
    this.restClient = restClient;
    this.batchSize = batchSize;
  }

  @Override
  protected SessionMetadata execute() {
    int statusCode = 200;
    try {
      if (restClient.isConnected()) {

        if (cache.isEmpty()) {
          LOGGER.trace("{} queue is empty, skipping run.", cache.getName());
        } else {
          List<AnalyticsHttpEvent> events = new ArrayList<>();

          for (AnalyticsHttpEvent event = cache.poll(); events.size() < batchSize && event != null; event = cache.poll()) {
            events.add(event);
          }

          LOGGER.trace("Attempting to send {} events from {} to analytics ingest.", events.size(), cache.getName());

          try {
            statusCode = restClient.postHttpEvents(events);

            LOGGER.debug("{} events from {} were sent to analytics ingest.", events.size(), cache.getName());

          } catch (Exception e) {
            normalizedLogger.warn(SEND_ANALYTICS_DESCRIPTION, cache.getName(), e);
            cache.addAll(events);
            statusCode = 503;
          }
        }
      }

    } catch (Throwable t) {
      normalizedLogger.error(SEND_ANALYTICS_DESCRIPTION, cache.getName(), t);
      return new ErrorSessionMetadata();
    }
    return new ApiPlatformSessionMetadata(statusCode);
  }
}
