/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.extractor;

import static com.hazelcast.util.Preconditions.checkNotNull;
import static com.mulesoft.mule.runtime.gw.api.security.ClientAuthenticationBuilder.CLIENT_ID_KEY;
import static com.mulesoft.mule.runtime.gw.api.security.ClientAuthenticationBuilder.CLIENT_NAME_KEY;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.runtime.api.notification.EnrichedServerNotification;
import org.mule.runtime.api.security.Authentication;

import com.mulesoft.mule.runtime.gw.api.client.Client;

import java.util.Optional;

/**
 * Extracts information needed by analytics from the server notification
 */
public class EnrichedServerNotificationExtractor {

  private final EnrichedServerNotificationEventIdExtractor eventIdExtractor = new EnrichedServerNotificationEventIdExtractor();

  public String flow(EnrichedServerNotification notification) {
    checkNotNull(notification, "Notification cannot be null");
    return notification.getResourceIdentifier();
  }

  public String eventId(EnrichedServerNotification notification) {
    return eventIdExtractor.eventId(notification);
  }

  public Optional<Client> client(EnrichedServerNotification notification) {
    checkNotNull(notification, "Notification cannot be null");

    if (notification.getEvent().getAuthentication().isPresent()) {
      Authentication authentication = notification.getEvent().getAuthentication().get();

      if (authentication.getProperties() != null && authentication.getProperties().containsKey(CLIENT_ID_KEY)) {
        return of(new NoSecretClient((String) authentication.getProperties().get(CLIENT_ID_KEY),
                                     (String) authentication.getProperties().get(CLIENT_NAME_KEY)));
      }
    }

    return empty();
  }

  static class NoSecretClient extends Client {

    private static final long serialVersionUID = 4049302420734670090L;
    private final String id;
    private final String name;

    NoSecretClient(String id, String name) {
      this.id = id;
      this.name = name;
    }

    @Override
    public String id() {
      return id;
    }

    @Override
    public String secret() {
      return null;
    }

    @Override
    public String name() {
      return name;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }
      if (!super.equals(o)) {
        return false;
      }

      NoSecretClient that = (NoSecretClient) o;

      if (id != null ? !id.equals(that.id) : that.id != null) {
        return false;
      }
      return name != null ? name.equals(that.name) : that.name == null;
    }

    @Override
    public int hashCode() {
      int result = super.hashCode();
      result = 31 * result + (id != null ? id.hashCode() : 0);
      result = 31 * result + (name != null ? name.hashCode() : 0);
      return result;
    }
  }
}
