/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.extractor;

import static com.google.common.net.InetAddresses.isInetAddress;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.substringBefore;
import static org.mule.runtime.http.api.HttpHeaders.Names.X_FORWARDED_FOR;

import org.mule.runtime.api.util.MultiMap;

import com.mulesoft.mule.runtime.gw.analytics.model.HttpRequestAttributes;

import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DefaultClientIpExtractor implements ClientIpExtractor {

  private static final Pattern REMOTE_ADDRESS_PATTERN = Pattern.compile("([^/]*/+)?\\[?([^]]*)]?:(.*)");
  private static final String IP_SEPARATOR = ",";
  private static final String X_FORWARDED_FOR_LOWER = X_FORWARDED_FOR.toLowerCase();

  @Override
  public String extractClientIp(HttpRequestAttributes httpRequestAttributes) {
    return extractXForwardedIp(httpRequestAttributes.getHeaders())
        .orElse(extractRemoteAddressIp(httpRequestAttributes.getRemoteAddress()));
  }

  private String extractRemoteAddressIp(String remoteAddress) {
    if (remoteAddress != null) {
      Matcher ipMatcher = REMOTE_ADDRESS_PATTERN.matcher(remoteAddress);
      return ipMatcher.find() ? ipMatcher.group(2) : null;
    }

    return null;
  }

  private Optional<String> extractXForwardedIp(MultiMap<String, String> headers) {
    return ofNullable(headers.get(X_FORWARDED_FOR_LOWER)).flatMap(header -> {
      String ip = header.contains(IP_SEPARATOR) ? substringBefore(header, IP_SEPARATOR) : header;
      return isInetAddress(ip) ? of(ip) : empty();
    });
  }
}
