/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.cache;

import static java.util.Objects.requireNonNull;

import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueueFactory;

import java.util.Queue;
import java.util.function.Function;

import org.mapdb.DB;

public class AnalyticsEventCacheFactory {

  private final SizeLimitedQueueFactory factory;

  private final DB regularEventsDb;
  private final DB violationsEventsDb;

  public AnalyticsEventCacheFactory(DB regularEventsDb, DB violationsEventsDb) {
    this.regularEventsDb = regularEventsDb;
    this.violationsEventsDb = violationsEventsDb;
    this.factory = new SizeLimitedQueueFactory();
  }

  /**
   * Creates a Cache which internally uses a simple {@link com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueue}, with the provided
   * name and capacity
   * 
   * @param name name of the cache
   * @param cacheCapacity max entries of the cache
   * @return the newly created {@link AnalyticsEventCache}
   */
  public AnalyticsEventCache createRegularEventCache(String name, int cacheCapacity) {
    Queue<AnalyticsHttpEvent> regularQueue = regularEventsDb.getQueue(name);
    return new AnalyticsEventCache(factory.createCircularQueue(regularQueue, cacheCapacity), name);
  }

  /**
   * Creates a Cache which internally uses a {@link com.mulesoft.mule.runtime.gw.queue.SizeLimitedPartitionedQueue}, with the
   * provided name, capacity and partition size
   *
   * @param name name of the cache
   * @param cacheCapacity max entries of the cache
   * @param violationThreshold value used for the size of the partition
   * @return the newly created {@link AnalyticsEventCache}
   */
  public AnalyticsEventCache createPolicyViolationsCache(String name, int cacheCapacity, int violationThreshold) {
    Queue<AnalyticsHttpEvent> violationsQueue = violationsEventsDb.getQueue(name);

    Function<AnalyticsHttpEvent, String> partitioner = event -> {
      requireNonNull(event.getPolicyViolation(), "Policy violation must be present in the Analytics event for this queue");
      return event.getPolicyViolation().getPolicyId().toString();
    };

    return new AnalyticsEventCache(factory.createCircularPartitionedQueue(violationsQueue,
                                                                          cacheCapacity,
                                                                          violationThreshold,
                                                                          partitioner),
                                   name);
  }

}
