/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.model;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_EXCHANGE_NAME;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_INSTANCE_NAME;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_PRODUCT_VERSION;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_VERSION;
import static com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolationOutcome.ERROR;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static org.apache.http.HttpHeaders.VIA;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.notification.ConnectorMessageNotification.MESSAGE_ERROR_RESPONSE;
import static org.mule.runtime.api.notification.ConnectorMessageNotification.MESSAGE_RESPONSE;
import static org.mule.runtime.http.api.HttpConstants.Method.POST;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.USER_AGENT;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Date;
import java.util.OptionalLong;

import org.joda.time.format.ISODateTimeFormat;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

import org.mule.extension.http.api.HttpRequestAttributes;
import org.mule.extension.http.api.HttpResponseAttributes;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.notification.IntegerAction;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.api.util.Pair;
import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.analytics.extractor.DefaultClientIpExtractor;
import com.mulesoft.mule.runtime.gw.analytics.feedbackloop.EdgeRequestDetector;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolation;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;

@RunWith(MockitoJUnitRunner.class)
public class AnalyticsHttpEventBuilderTestCase extends AbstractMuleTestCase {

  private static final String ORG_ID = "orgId";
  private static final String EVENT_ID = "eventId";
  private static final String CLIENT_ID = "clientId";
  private static final String CLIENT_NAME = "clientName";
  private static final String POLICY_NAME = "policyName";
  private static final String REQUEST_URI = "requestUri";
  private static final String REMOTE_IP = "127.0.0.1";
  private static final String REMOTE_ADDRESS = format("bar/%s:5701", REMOTE_IP);
  private static final String A_USER_AGENT = "aUserAgent";
  private static final IntegerAction RESPONSE_ACTION = new IntegerAction(MESSAGE_RESPONSE);
  private static final IntegerAction ERROR_RESPONSE_ACTION = new IntegerAction(MESSAGE_ERROR_RESPONSE);

  @Mock
  private Api api;

  @Mock
  private Client client;

  @Mock
  private TrackingInfo trackingInfo;

  @Mock
  private PolicyViolation policyViolation;

  private TypedValue<Object> requestAttributes;
  private TypedValue<Object> responseAttributes;

  private AnalyticsEventBuilder eventBuilder;

  @Before
  public void setUp() {
    eventBuilder = new AnalyticsHttpEventBuilder(new EdgeRequestDetector(), new HttpRequestAttributesFactory(),
                                                 new HttpResponseAttributesFactory(), new DefaultClientIpExtractor());

    when(trackingInfo.getInstanceName()).thenReturn(API_INSTANCE_NAME);
    when(trackingInfo.getVersion()).thenReturn(API_VERSION);
    when(trackingInfo.getLegacyApiIdentifier()).thenReturn(1);
    when(trackingInfo.getExchangeAssetName()).thenReturn(API_EXCHANGE_NAME);
    when(trackingInfo.getProductVersion()).thenReturn(API_PRODUCT_VERSION);
    when(trackingInfo.getOrganizationId()).thenReturn(ORG_ID);

    when(api.getTrackingInfo()).thenReturn(trackingInfo);
    when(api.getKey()).thenReturn(API_KEY);

    when(client.id()).thenReturn(CLIENT_ID);
    when(client.name()).thenReturn(CLIENT_NAME);

    when(policyViolation.getPolicyId()).thenReturn("10");
    when(policyViolation.getPolicyName()).thenReturn(POLICY_NAME);
    when(policyViolation.getOutcome()).thenReturn(ERROR);

    requestAttributes = TypedValue.of(httpRequestAttributes(new Pair<>(CONTENT_LENGTH, "100"),
                                                            new Pair<>(USER_AGENT, A_USER_AGENT)));

    responseAttributes = TypedValue.of(httpResponseAttributes(new Pair<>(CONTENT_LENGTH, "200")));
  }

  @Test
  public void buildEvent() throws UnknownHostException {
    Date receivedDate = new Date();
    Date repliedDate = new Date();
    eventBuilder.withApi(api);
    eventBuilder.withClient(client);
    eventBuilder.withPolicyViolation(policyViolation);
    eventBuilder.withId(EVENT_ID);
    eventBuilder.withRequest(OptionalLong.of(100), requestAttributes);
    eventBuilder.withResponse(OptionalLong.of(200), responseAttributes, RESPONSE_ACTION);
    eventBuilder.withReceivedTimestamp(receivedDate.getTime());
    eventBuilder.withRepliedTimestamp(repliedDate.getTime());

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getApiId(), is(1));
    assertThat(event.getApiName(), is(API_EXCHANGE_NAME));
    assertThat(event.getInstanceName(), is(API_INSTANCE_NAME));
    assertThat(event.getApiVersion(), is(API_PRODUCT_VERSION));
    assertThat(event.getApiVersionId(), is(API_KEY.id()));
    assertThat(event.getApplicationName(), is(CLIENT_NAME));
    assertThat(event.getClientId(), is(CLIENT_ID));
    assertThat(event.getClientIp(), is(REMOTE_IP));
    assertThat(event.getEventId(), is(EVENT_ID));
    assertThat(event.getHostId(), is(InetAddress.getLocalHost().getHostName()));
    assertThat(event.getOrgId(), is(ORG_ID));
    assertThat(event.getPath(), is(REQUEST_URI));
    assertThat(event.getPolicyViolation().getOutcome(), is(ERROR));
    assertThat(event.getPolicyViolation().getPolicyId(), is("10"));
    assertThat(event.getPolicyViolation().getPolicyName(), is(POLICY_NAME));
    assertThat(event.getReceivedTs(), is(ISODateTimeFormat.dateTime().print(receivedDate.getTime())));
    assertThat(event.getRepliedTs(), is(ISODateTimeFormat.dateTime().print(repliedDate.getTime())));
    assertThat(event.getRequestBytes(), is(100));
    assertThat(event.getResponseBytes(), is(200));
    assertThat(event.getStatusCode(), is(515));
    assertThat(event.getTransactionId(), nullValue());
    assertThat(event.getUserAgent(), is(A_USER_AGENT));
    assertThat(event.getVerb(), is(POST.name()));
    assertThat(event.getDeploymentType(), nullValue());
  }

  @Test
  public void buildEventWithoutContentLength() {
    eventBuilder.withRequest(OptionalLong.empty(), TypedValue.of(httpRequestAttributes()));
    eventBuilder.withResponse(OptionalLong.empty(), TypedValue.of(httpResponseAttributes()), RESPONSE_ACTION);

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getRequestBytes(), is(-1));
    assertThat(event.getResponseBytes(), is(-1));
  }

  @Test
  public void buildEventWithoutResponseAttributesButWithoutError() {
    eventBuilder.withResponse(OptionalLong.empty(), TypedValue.of(httpRequestAttributes()), RESPONSE_ACTION);

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getStatusCode(), is(200));
  }

  @Test
  public void buildEventWithoutResponseAttributesButWithError() {
    eventBuilder.withResponse(OptionalLong.empty(), TypedValue.of(httpRequestAttributes()), ERROR_RESPONSE_ACTION);

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getStatusCode(), is(500));
  }

  @Test
  public void buildEventWithContentLengthFromHeader() {
    eventBuilder.withRequest(OptionalLong.empty(), TypedValue.of(httpRequestAttributes(new Pair<>(CONTENT_LENGTH, "3"))));
    eventBuilder.withResponse(OptionalLong.empty(), TypedValue.of(httpResponseAttributes(new Pair<>(CONTENT_LENGTH, "2"))),
                              RESPONSE_ACTION);

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getRequestBytes(), is(3));
    assertThat(event.getResponseBytes(), is(2));
  }

  @Test
  public void buildEventWithContentLengthFromHeaderCaseInsensitive() {
    eventBuilder.withRequest(OptionalLong.empty(),
                             TypedValue.of(httpRequestAttributes(new Pair<>(CONTENT_LENGTH.toUpperCase(), "3"))));
    eventBuilder.withResponse(OptionalLong.empty(),
                              TypedValue.of(httpResponseAttributes(new Pair<>(CONTENT_LENGTH.toUpperCase(), "2"))),
                              RESPONSE_ACTION);

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getRequestBytes(), is(3));
    assertThat(event.getResponseBytes(), is(2));
  }

  @Test
  public void buildEventWithContentLengthFromHeaderInvalid() {
    eventBuilder.withRequest(OptionalLong.empty(), TypedValue.of(httpRequestAttributes(new Pair<>(CONTENT_LENGTH, "A"))));
    eventBuilder.withResponse(OptionalLong.empty(), TypedValue.of(httpResponseAttributes(new Pair<>(CONTENT_LENGTH, "A"))),
                              RESPONSE_ACTION);

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getRequestBytes(), is(-1));
    assertThat(event.getResponseBytes(), is(-1));
  }

  @Test
  public void buildEventWithContentLengthIntOverflow() {
    eventBuilder.withRequest(OptionalLong.of(Long.MAX_VALUE), TypedValue.of(httpRequestAttributes()));
    eventBuilder.withResponse(OptionalLong.of(Long.MAX_VALUE), TypedValue.of(httpResponseAttributes()), RESPONSE_ACTION);

    AnalyticsHttpEvent event = eventBuilder.build();

    assertThat(event.getRequestBytes(), is(-1));
    assertThat(event.getResponseBytes(), is(-1));
  }

  @Test
  public void buildEventWithPerApiOrgId() {
    String apiOrgId = "api-org-id";
    when(trackingInfo.getOrganizationId()).thenReturn(apiOrgId);

    eventBuilder.withApi(api);

    AnalyticsHttpEvent event = eventBuilder.build();
    assertThat(event.getOrgId(), is(apiOrgId));
  }

  @Test
  public void isEdgeRequest() {
    requestAttributes = TypedValue.of(httpRequestAttributes(new Pair<>(VIA, "EDGE/1.1 MuleSoft")));
    eventBuilder.withRequest(OptionalLong.empty(), TypedValue.of(requestAttributes));

    assertThat(eventBuilder.isEdgeRequest(), is(true));
  }

  @Test
  public void isEdgeRequestDifferentCasing() {
    requestAttributes = TypedValue.of(httpRequestAttributes(new Pair<>(VIA.toUpperCase(), "EDGE/1.1 MuleSoft")));
    eventBuilder.withRequest(OptionalLong.empty(), TypedValue.of(requestAttributes));

    assertThat(eventBuilder.isEdgeRequest(), is(true));
  }

  @Test
  public void isEdgeRequestIsLazilyCalculated() {
    EdgeRequestDetector edgeRequestDetector = mock(EdgeRequestDetector.class);
    eventBuilder = new AnalyticsHttpEventBuilder(edgeRequestDetector, new HttpRequestAttributesFactory(),
                                                 new HttpResponseAttributesFactory(), new DefaultClientIpExtractor());
    requestAttributes = TypedValue.of(httpRequestAttributes(new Pair<>(VIA.toUpperCase(), "EDGE/1.1 MuleSoft")));

    eventBuilder.withRequest(OptionalLong.empty(), TypedValue.of(requestAttributes));

    verify(edgeRequestDetector, never()).isEdgeRequest(any());

    eventBuilder.isEdgeRequest();

    verify(edgeRequestDetector, times(1)).isEdgeRequest(any());
  }

  private HttpRequestAttributes httpRequestAttributes(Pair<String, String>... headerTuples) {
    MultiMap<String, String> headers = new MultiMap<>();
    asList(headerTuples).forEach(header -> headers.put(header.getFirst(), header.getSecond()));

    return new HttpRequestAttributes(headers, null, null, null, null, POST.name(), null, REQUEST_URI, null, null, null,
                                     REMOTE_ADDRESS, null);
  }

  private HttpResponseAttributes httpResponseAttributes(Pair<String, String>... headerTuples) {
    MultiMap<String, String> headers = new MultiMap<>();
    asList(headerTuples).forEach(header -> headers.put(header.getFirst(), header.getSecond()));

    return new HttpResponseAttributes(515, null, headers);
  }
}
