/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.cache;

import java.util.Collection;
import java.util.Iterator;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;

import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueue;

/**
 * In order to maintain the interface with Mule's Agent, this cache has to be a {@link BlockingQueue}. Nevertheless, only the
 * non-blocking part of the cache is implemented. The blocking methods throw a {@link UnsupportedOperationException}.
 */
public class AnalyticsEventCache implements Queue<AnalyticsHttpEvent> {

  private final SizeLimitedQueue<AnalyticsHttpEvent> queue;
  private final String queueName;

  public AnalyticsEventCache(SizeLimitedQueue<AnalyticsHttpEvent> queue, String queueName) {
    this.queue = queue;
    this.queueName = queueName;
  }

  public String getName() {
    return queueName;
  }

  /**
   * Method synchronized to avoid NPE when retrieving from the queue under high load.
   */
  private synchronized boolean doAdd(AnalyticsHttpEvent event) {
    return queue.add(event);
  }

  private Optional<AnalyticsHttpEvent> doRetrieve() {
    return queue.retrieve();
  }

  @Override
  public int size() {
    return queue.size();
  }

  @Override
  public boolean isEmpty() {
    return queue.isEmpty();
  }

  @Override
  public synchronized boolean add(AnalyticsHttpEvent event) {
    boolean added = doAdd(event);

    if (!added) {
      throw new IllegalArgumentException("Element could not be added to the queue");
    }

    return true;
  }

  @Override
  public boolean offer(AnalyticsHttpEvent httpEvent) {
    return doAdd(httpEvent);
  }

  @Override
  public AnalyticsHttpEvent remove() {
    return doRetrieve().get();
  }

  @Override
  public AnalyticsHttpEvent poll() {
    return doRetrieve().orElse(null);
  }

  @Override
  public AnalyticsHttpEvent element() {
    return queue.peek().get();
  }

  @Override
  public AnalyticsHttpEvent peek() {
    return queue.peek().orElse(null);
  }

  @Override
  public boolean remove(Object o) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean containsAll(Collection<?> c) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean addAll(Collection<? extends AnalyticsHttpEvent> events) {
    boolean anyAdded = false;

    for (AnalyticsHttpEvent event : events) {
      anyAdded = doAdd(event) || anyAdded;
    }

    return anyAdded;
  }

  @Override
  public boolean removeAll(Collection<?> c) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean retainAll(Collection<?> c) {
    throw new UnsupportedOperationException();
  }

  @Override
  public void clear() {
    queue.clear();
  }

  @Override
  public boolean contains(Object o) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Iterator<AnalyticsHttpEvent> iterator() {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object[] toArray() {
    throw new UnsupportedOperationException();
  }

  @Override
  public <T> T[] toArray(T[] a) {
    throw new UnsupportedOperationException();
  }

}
