/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.cache;

import static java.util.Optional.ofNullable;
import static org.apache.commons.io.FileUtils.deleteQuietly;

import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.mapdb.DB;
import org.mapdb.DBMaker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AnalyticsEventCacheManager {

  private static final Logger LOGGER = LoggerFactory.getLogger(AnalyticsEventCacheManager.class);

  private static final String CACHE_DATA_FILE_EXT = ".p";

  private static final String ANALYTICS_QUEUE_NAME = "events-queue";
  private static final String ANALYTICS_AGENT_QUEUE_NAME = "agent-events-queue";

  private final AnalyticsConfiguration configuration;

  private AnalyticsEventCaches cloudCaches;
  private AnalyticsEventCaches agentCaches;

  private List<AnalyticsEventCache> availableRegularCaches;
  private List<AnalyticsEventCache> availableViolationsCaches;

  private AnalyticsEventCacheFactory eventCacheFactory;

  private DB regularEventsDb;
  private DB violationsEventsDb;

  public AnalyticsEventCacheManager(AnalyticsConfiguration configuration) {
    this.configuration = configuration;

    this.availableRegularCaches = new ArrayList<>();
    this.availableViolationsCaches = new ArrayList<>();
  }

  public void initialise() {
    regularEventsDb = createDb(configuration.getAnalyticsCacheFile(), "Regular events queue");
    violationsEventsDb = createDb(configuration.getAnalyticsPolicyViolationsCacheFile(), "Violations events queue");
    eventCacheFactory = new AnalyticsEventCacheFactory(regularEventsDb, violationsEventsDb);

    if (configuration.eventsToCloudEnabled()) {
      cloudCaches = createEventCaches(ANALYTICS_QUEUE_NAME);
    }

    // in service-mesh only agent pushes events to the queues and only gw pulls them
    if (configuration.isServiceMesh()) {
      agentCaches = cloudCaches;
    } else if (configuration.eventsThroughAgentEnabled()) {
      agentCaches = createEventCaches(ANALYTICS_AGENT_QUEUE_NAME);
    }
  }

  public void dispose() {
    if (regularEventsDb != null) {
      regularEventsDb.close();
    }

    if (violationsEventsDb != null) {
      violationsEventsDb.close();
    }
  }

  public Optional<AnalyticsEventCaches> getCloudCaches() {
    return ofNullable(cloudCaches);
  }

  public Optional<AnalyticsEventCaches> getAgentCaches() {
    return ofNullable(agentCaches);
  }

  public List<AnalyticsEventCache> getAvailableRegularCaches() {
    return availableRegularCaches;
  }

  public List<AnalyticsEventCache> getAvailablePolicyViolationsCache() {
    return availableViolationsCaches;
  }

  private AnalyticsEventCaches createEventCaches(String queueName) {
    AnalyticsEventCache regularEventCache =
        eventCacheFactory.createRegularEventCache(queueName, configuration.getCacheCapacity());
    AnalyticsEventCache policyViolationCache =
        eventCacheFactory.createPolicyViolationsCache(queueName,
                                                      configuration.getPolicyViolationsCacheCapacity(),
                                                      configuration.getPolicyViolationThreshold());

    availableRegularCaches.add(regularEventCache);
    availableViolationsCaches.add(policyViolationCache);

    return new AnalyticsEventCaches(regularEventCache, policyViolationCache);
  }

  private DB createDb(File cacheFile, String cacheName) {
    DB db;

    try {
      db = makeDB(cacheFile);
    } catch (Throwable throwable) {
      LOGGER.warn("There was an error attempting to open {}. Resetting it to its factory settings. {}", cacheName, throwable);

      deleteQuietly(cacheFile);
      deleteQuietly(new File(cacheFile.getAbsolutePath() + CACHE_DATA_FILE_EXT));

      db = makeDB(cacheFile);
    }

    return db;
  }

  private DB makeDB(File file) {
    return DBMaker.newFileDB(file)
        .transactionDisable()
        .mmapFileEnable()
        .asyncWriteEnable()
        .closeOnJvmShutdown()
        .deleteFilesAfterClose()
        .make();
  }
}
