/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.model;

import static com.mulesoft.mule.runtime.gw.api.analytics.RequestDisposition.BLOCKED;
import static com.mulesoft.mule.runtime.gw.api.analytics.RequestDisposition.PROCESSED;
import static java.lang.Integer.parseInt;
import static java.lang.Math.toIntExact;
import static org.mule.runtime.api.notification.ConnectorMessageNotification.MESSAGE_ERROR_RESPONSE;
import static org.mule.runtime.api.notification.ConnectorMessageNotification.MESSAGE_RESPONSE;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.USER_AGENT;
import static org.mule.runtime.http.api.HttpHeaders.Names.VIA;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Optional;
import java.util.OptionalLong;

import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.notification.IntegerAction;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.MultiMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mulesoft.mule.runtime.gw.analytics.extractor.ClientIpExtractor;
import com.mulesoft.mule.runtime.gw.analytics.feedbackloop.EdgeRequestDetector;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent.Builder;
import com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolation;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.model.Api;

public class AnalyticsHttpEventBuilder implements AnalyticsEventBuilder {

  private static final Logger LOGGER = LoggerFactory.getLogger(AnalyticsHttpEventBuilder.class);

  private static final DateTimeFormatter DATE_TIME_FORMATTER = ISODateTimeFormat.dateTime();

  private static final String CONTENT_LENGTH_LOWER = CONTENT_LENGTH.toLowerCase();
  private static final String USER_AGENT_LOWER = USER_AGENT.toLowerCase();
  private static final String VIA_HEADER = VIA.toLowerCase();

  private static String hostname = getHostname();

  private final Builder httpEvent;

  private final EdgeRequestDetector edgeRequestDetector;
  private final HttpRequestAttributesFactory httpRequestAttributesFactory;
  private final HttpResponseAttributesFactory httpResponseAttributesFactory;
  private final ClientIpExtractor clientIpExtractor;

  private String apiDescription;

  private LazyValue<Boolean> isEdgeRequest;

  public AnalyticsHttpEventBuilder(EdgeRequestDetector edgeRequestDetector,
                                   HttpRequestAttributesFactory httpRequestAttributesFactory,
                                   HttpResponseAttributesFactory httpResponseAttributesFactory,
                                   ClientIpExtractor clientIpExtractor) {
    this.edgeRequestDetector = edgeRequestDetector;
    this.httpRequestAttributesFactory = httpRequestAttributesFactory;
    this.httpResponseAttributesFactory = httpResponseAttributesFactory;
    this.clientIpExtractor = clientIpExtractor;

    this.httpEvent = AnalyticsHttpEvent.builder();

    this.httpEvent.withRequestDisposition(PROCESSED);
    this.httpEvent.withHostId(hostname);
  }

  private static String getHostname() {
    try {
      return InetAddress.getLocalHost().getHostName();
    } catch (UnknownHostException e) {
      LOGGER.error("Local hostname could not be retrieved. Analytics events will not include it", e);
    }

    return null;
  }

  @Override
  public AnalyticsHttpEvent build() {
    return httpEvent.build();
  }

  @Override
  public AnalyticsHttpEventBuilder withApi(Api api) {
    apiDescription = api.toString();
    httpEvent.withApiVersionId(api.getKey().id());
    httpEvent.withApiName(api.getTrackingInfo().getExchangeAssetName());
    httpEvent.withApiVersion(api.getTrackingInfo().getProductVersion());
    httpEvent.withInstanceName(api.getTrackingInfo().getInstanceName());
    httpEvent.withApiId(api.getTrackingInfo().getLegacyApiIdentifier());
    httpEvent.withOrgId(api.getTrackingInfo().getOrganizationId());
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withRequest(OptionalLong payloadLength, TypedValue<Object> requestAttributes) {
    httpRequestAttributesFactory.from(requestAttributes).ifPresent(httpRequestAttributes -> {
      httpEvent.withUserAgent(httpRequestAttributes.getHeaders().get(USER_AGENT_LOWER));
      httpEvent.withRequestBytes(payloadBytes(payloadLength, httpRequestAttributes.getHeaders()));
      httpEvent.withPath(httpRequestAttributes.getRequestUri());
      httpEvent.withVerb(httpRequestAttributes.getMethod());
      httpEvent.withClientIp(clientIpExtractor.extractClientIp(httpRequestAttributes));
      isEdgeRequest =
          new LazyValue<>(() -> edgeRequestDetector.isEdgeRequest(httpRequestAttributes.getHeaders().getAll(VIA_HEADER)));
    });
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withId(String id) {
    httpEvent.withEventId(id);
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withMuleAppName(String muleAppName) {
    httpEvent.withMuleAppName(muleAppName);
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withFlowName(String flowName) {
    httpEvent.withFlowName(flowName);
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withReceivedTimestamp(long timestamp) {
    httpEvent.withReceivedTs(DATE_TIME_FORMATTER.print(timestamp));
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withRepliedTimestamp(long timestamp) {
    httpEvent.withRepliedTs(DATE_TIME_FORMATTER.print(timestamp));
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withPolicyViolation(PolicyViolation policyViolation) {
    httpEvent.withPolicyViolation(policyViolation);
    httpEvent.withRequestDisposition(BLOCKED);
    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withResponse(OptionalLong payloadLength, TypedValue<Object> attributes,
                                                IntegerAction action) {
    Optional<HttpResponseAttributes> responseAttributes = httpResponseAttributesFactory.from(attributes);

    if (responseAttributes.isPresent()) {
      HttpResponseAttributes httpResponseAttributes = responseAttributes.get();
      httpEvent.withStatusCode(httpResponseAttributes.getStatusCode());
      httpEvent.withResponseBytes(payloadBytes(payloadLength, httpResponseAttributes.getHeaders()));
    } else if (new IntegerAction(MESSAGE_RESPONSE).equals(action)) {
      httpEvent.withStatusCode(OK.getStatusCode());
    } else if (new IntegerAction(MESSAGE_ERROR_RESPONSE).equals(action)) {
      httpEvent.withStatusCode(INTERNAL_SERVER_ERROR.getStatusCode());
    }

    return this;
  }

  @Override
  public AnalyticsHttpEventBuilder withClient(Client client) {
    httpEvent.withClientId(client.id());
    httpEvent.withApplicationName(client.name());
    return this;
  }

  @Override
  public boolean isEdgeRequest() {
    return isEdgeRequest.get();
  }

  @Override
  public String getApiDescription() {
    return apiDescription;
  }

  private int payloadBytes(OptionalLong payloadLength, MultiMap<String, String> headers) {
    try {
      if (payloadLength.isPresent()) {
        return toIntExact(payloadLength.getAsLong());
      } else if (headers.containsKey(CONTENT_LENGTH_LOWER)) {
        return parseInt(headers.get(CONTENT_LENGTH_LOWER));
      }
    } catch (ArithmeticException | NumberFormatException e) {
      LOGGER.debug("Unexpected error when trying to calculate payload size: {}", e.getMessage());
    }

    return -1;
  }

}
