/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.notification;

import static com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolation.builder;
import static org.mule.runtime.api.notification.PolicyNotification.AFTER_NEXT;
import static org.mule.runtime.api.notification.PolicyNotification.PROCESS_END;

import org.mule.runtime.api.notification.IntegerAction;
import org.mule.runtime.api.notification.PolicyNotification;
import org.mule.runtime.api.notification.PolicyNotificationListener;

import com.mulesoft.mule.runtime.gw.analytics.extractor.EnrichedServerNotificationEventIdExtractor;
import com.mulesoft.mule.runtime.gw.analytics.model.HttpResponseAttributesFactory;
import com.mulesoft.mule.runtime.gw.analytics.model.HttpStatusCodeMatcher;
import com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolation;
import com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolationOutcome;
import com.mulesoft.mule.runtime.gw.api.service.EventsCollectorService;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Notification listener in charge of detecting if a policy violation happened during a particular policy execution
 */
public class AnalyticsPolicyNotificationListener implements PolicyNotificationListener<PolicyNotification> {

  private static final Logger LOGGER = LoggerFactory.getLogger(AnalyticsPolicyNotificationListener.class);

  private final String policyName;
  private final String policyId;
  private final String violationCategory;
  private final HttpStatusCodeMatcher statusCodeMatcher;
  private final EnrichedServerNotificationEventIdExtractor eventIdExtractor = new EnrichedServerNotificationEventIdExtractor();
  private Map<String, Integer> statusCodePerEvent = new ConcurrentHashMap<>();
  private EventsCollectorService eventsCollectorService;
  private final HttpResponseAttributesFactory httpResponseAttributesFactory = new HttpResponseAttributesFactory();

  public AnalyticsPolicyNotificationListener(EventsCollectorService eventsCollectorService, String policyId, String policyName,
                                             String violationCategory) {
    this.policyId = policyId;
    this.policyName = policyName;
    this.statusCodeMatcher = new HttpStatusCodeMatcher();
    this.eventsCollectorService = eventsCollectorService;
    this.violationCategory = violationCategory;

  }

  @Override
  public void onNotification(PolicyNotification notification) {
    String eventId = eventIdExtractor.eventId(notification);

    if (isAfterNext(notification)) {
      httpResponseAttributesFactory.from(notification.getEvent().getMessage().getAttributes())
          .ifPresent(attributes -> statusCodePerEvent.put(eventId, attributes.getStatusCode()));
    } else if (isProcessEnd(notification)) {
      Integer previousStatusCode = statusCodePerEvent.remove(eventId);

      httpResponseAttributesFactory.from(notification.getEvent().getMessage().getAttributes()).ifPresent(attributes -> {
        Integer currentStatusCode = attributes.getStatusCode();
        if (isPolicyViolation(previousStatusCode, currentStatusCode)) {
          handleViolation(eventId, policyId, policyName);
        } else if (eventsCollectorService.getPolicyViolation(eventId).isPresent() && currentStatusCode < 400) {
          eventsCollectorService.removePolicyViolation(eventId);
        }
      });
    }
  }

  private void handleViolation(String eventId, String policyId, String policyName) {
    LOGGER.debug("Processing Policy Violation Notification for '{}'", policyName);

    PolicyViolation policyViolation =
        builder()
            .withPolicyId(policyId)
            .withPolicyName(policyName)
            .withOutcome(PolicyViolationOutcome.VIOLATION)
            .withCategory(violationCategory)
            .build();

    eventsCollectorService.addPolicyViolation(eventId, policyViolation);
  }

  /**
   * Is a violation if after policy execution the status code is greater than 400 and the previous policy/flow ended its execution
   * with a status code lesser than 400
   */
  private boolean isPolicyViolation(Integer previousStatusCode, Integer currentstatusCode) {
    return (previousStatusCode == null || !statusCodeMatcher.isError(previousStatusCode))
        && statusCodeMatcher.isError(currentstatusCode);
  }

  private boolean isAfterNext(PolicyNotification notification) {
    return new IntegerAction(AFTER_NEXT).equals(notification.getAction());
  }

  private boolean isProcessEnd(PolicyNotification notification) {
    return new IntegerAction(PROCESS_END).equals(notification.getAction());
  }

}
