/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.cache;

import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.container.api.MuleFoldersUtil.getExecutionFolder;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedPartitionedQueue;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueue;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import java.io.File;
import java.util.Optional;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class AnalyticsEventCacheManagerTestCase extends AbstractMuleTestCase {

  @Rule
  public SystemPropertyTemporaryFolder temporaryFolder = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  private AnalyticsConfiguration configuration;

  private AnalyticsEventCacheManager manager;

  @Before
  public void setUp() {
    configuration = mock(AnalyticsConfiguration.class);
    when(configuration.getCacheCapacity()).thenReturn(3);
    when(configuration.getPolicyViolationsCacheCapacity()).thenReturn(2);
    when(configuration.getPolicyViolationThreshold()).thenReturn(1);
    when(configuration.getAnalyticsCacheFile()).thenCallRealMethod();
    when(configuration.getAnalyticsPolicyViolationsCacheFile()).thenCallRealMethod();

    manager = new AnalyticsEventCacheManager(configuration);
  }

  @Test
  public void initialiseOnlyCloud() {
    when(configuration.eventsToCloudEnabled()).thenReturn(true);
    manager.initialise();

    Optional<AnalyticsEventCaches> cache = manager.getCloudCaches();

    assertThat(manager.getAgentCaches().isPresent(), is(false));
    assertThat(cache.isPresent(), is(true));
    assertCaches(cache.get());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics.db").exists());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics-policy-violations.db").exists());
  }

  @Test
  public void initialiseOnlyAgent() {
    when(configuration.eventsThroughAgentEnabled()).thenReturn(true);
    manager.initialise();

    Optional<AnalyticsEventCaches> cache = manager.getAgentCaches();

    assertThat(manager.getCloudCaches().isPresent(), is(false));
    assertThat(cache.isPresent(), is(true));
    assertCaches(cache.get());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics.db").exists());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics-policy-violations.db").exists());
  }

  @Test
  public void initialiseBothCloudAndAgent() {
    when(configuration.eventsToCloudEnabled()).thenReturn(true);
    when(configuration.eventsThroughAgentEnabled()).thenReturn(true);
    manager.initialise();

    Optional<AnalyticsEventCaches> cloudCaches = manager.getCloudCaches();
    Optional<AnalyticsEventCaches> agentCaches = manager.getAgentCaches();

    assertThat(cloudCaches.isPresent(), is(true));
    assertThat(agentCaches.isPresent(), is(true));
    assertNotSame(agentCaches.get(), cloudCaches.get());
    assertCaches(cloudCaches.get());
    assertCaches(agentCaches.get());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics.db").exists());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics-policy-violations.db").exists());
  }

  @Test
  public void initialiseServiceMesh() {
    when(configuration.eventsToCloudEnabled()).thenReturn(true);
    when(configuration.isServiceMesh()).thenReturn(true);
    manager.initialise();

    Optional<AnalyticsEventCaches> cloudCaches = manager.getCloudCaches();
    Optional<AnalyticsEventCaches> agentCaches = manager.getAgentCaches();

    assertThat(cloudCaches.isPresent(), is(true));
    assertThat(agentCaches.isPresent(), is(true));
    assertSame(agentCaches.get(), cloudCaches.get());
    assertCaches(cloudCaches.get());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics.db").exists());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics-policy-violations.db").exists());
  }

  @Test
  public void initialiseNone() {
    manager.initialise();

    assertThat(manager.getCloudCaches().isPresent(), is(false));
    assertThat(manager.getAgentCaches().isPresent(), is(false));
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics.db").exists());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics-policy-violations.db").exists());
  }

  @Test
  public void dbDeletedAfterClose() {
    manager.initialise();

    assertTrue(new File(getExecutionFolder(), "api-platform-analytics.db").exists());
    assertTrue(new File(getExecutionFolder(), "api-platform-analytics-policy-violations.db").exists());

    manager.dispose();

    assertFalse(new File(getExecutionFolder(), "api-platform-analytics.db").exists());
    assertFalse(new File(getExecutionFolder(), "api-platform-analytics-policy-violations.db").exists());
  }

  private void assertCaches(AnalyticsEventCaches caches) {
    assertThat(queue(caches.getRegularEventsCache()), instanceOf(SizeLimitedQueue.class));
    assertThat(new Inspector(queue(caches.getRegularEventsCache())).read("limit"), is(3L));
    assertThat(queue(caches.getViolationsEventsCache()), instanceOf(SizeLimitedPartitionedQueue.class));
    assertThat(new Inspector(queue(caches.getViolationsEventsCache())).read("limit"), is(2L));
    assertThat(new Inspector(queue(caches.getViolationsEventsCache())).read("partitionLimit"), is(1L));
  }

  private SizeLimitedQueue queue(AnalyticsEventCache cache) {
    return new Inspector(cache).read("queue");
  }

}
