/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.runnable;

import static com.mulesoft.mule.runtime.gw.analytics.backoff.AnalytictsBackoffConfigurationSupplier.DISPERSION;
import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.BACKOFF;
import static com.mulesoft.mule.runtime.gw.api.time.period.Period.seconds;
import static com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration.ANALYTICS_OUTAGE_STATUS_CODES;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.stream.IntStream.range;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.io.IOException;
import java.util.Arrays;
import java.util.function.Consumer;
import java.util.function.Supplier;

import org.junit.Before;
import org.junit.Test;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

import com.mulesoft.anypoint.tests.scheduler.ObservableScheduledExecutorService;
import com.mulesoft.anypoint.tests.scheduler.observer.ScheduledTask;
import com.mulesoft.mule.runtime.gw.analytics.AnalyticsPollersManager;
import com.mulesoft.mule.runtime.gw.analytics.backoff.AnalytictsBackoffConfigurationSupplier;
import com.mulesoft.mule.runtime.gw.analytics.cache.AnalyticsEventCaches;
import com.mulesoft.mule.runtime.gw.api.time.period.Period;
import com.mulesoft.anypoint.backoff.BackoffTestCase;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.engine.BackoffSimulation;
import com.mulesoft.anypoint.backoff.function.dispersion.RangeDispersant;
import com.mulesoft.anypoint.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.anypoint.backoff.scheduler.factory.FixedExecutorBackoffSchedulerFactory;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

public class AnalyticsBackoffTestCase extends BackoffTestCase {

  @Before
  public void setUp() {
    super.setUp();
    clearProperty(BACKOFF);
    clearProperty(ANALYTICS_OUTAGE_STATUS_CODES);
  }

  @Test
  public void analyticsNoErrorsRemainsStable() {
    noErrorsRemainsStable(this::getAnalyticsInitialDelay, AnalyticsPollersManager::scheduleAnalyticsRunnable);
  }

  @Test
  public void policyViolationsNoErrorsRemainsStable() {
    noErrorsRemainsStable(this::getPolicyViolationsInitialDelay, AnalyticsPollersManager::schedulePolicyViolationsRunnable);
  }

  @Test
  public void analyticsBackoffBackon() {
    backoffBackon(this::getAnalyticsInitialDelay, AnalyticsPollersManager::scheduleAnalyticsRunnable);
  }

  @Test
  public void policyViolationsBackoffBackon() {
    backoffBackon(this::getPolicyViolationsInitialDelay, AnalyticsPollersManager::schedulePolicyViolationsRunnable);
  }

  @Test
  public void backoffDisabled() {
    setProperty(BACKOFF, "false");

    int iterations = 50;
    BackoffSimulation simulation = backoffSimulation(this::getAnalyticsInitialDelay)
        .off(0, iterations, 503)
        .disabled()
        .simulate(iterations);

    assertExecutionMatchesSimulation(AnalyticsPollersManager::scheduleAnalyticsRunnable, simulation, iterations);
  }

  @Test
  public void defaultOutageStatusCodes() {
    BackoffConfiguration configuration = new AnalytictsBackoffConfigurationSupplier().get();
    assertThat(configuration.statusCodes(), is(Arrays.asList(429, 500, 501, 502, 503, 504)));
  }

  @Test
  public void customOutageStatusCodes() {
    setProperty(ANALYTICS_OUTAGE_STATUS_CODES, "401, 501");
    BackoffConfiguration configuration = new AnalytictsBackoffConfigurationSupplier().get();
    assertThat(configuration.statusCodes(), is(Arrays.asList(401, 501)));
  }

  private void noErrorsRemainsStable(Supplier<Period> initialDelay, Consumer<AnalyticsPollersManager> startScheduler) {
    int simulationIterations = 100;
    BackoffSimulation simulation = backoffSimulation(initialDelay).simulate(simulationIterations);
    assertExecutionMatchesSimulation(startScheduler, simulation, simulationIterations);
  }

  private void backoffBackon(Supplier<Period> initialDelay, Consumer<AnalyticsPollersManager> startScheduler) {
    int iterations = 10;
    BackoffSimulation simulation = backoffSimulation(initialDelay)
        .off(0, 5, 503)
        .simulate(iterations);
    assertExecutionMatchesSimulation(startScheduler, simulation, iterations);
  }

  private BackoffSimulation backoffSimulation(Supplier<Period> initialDelay) {
    return new BackoffSimulation(initialDelay.get(), getFrequency(),
                                 withNoDispersion(new AnalytictsBackoffConfigurationSupplier().get()));
  }

  private Period getAnalyticsInitialDelay() {
    return seconds(5);
  }

  private Period getPolicyViolationsInitialDelay() {
    return seconds(7);
  }

  private Period getFrequency() {
    return seconds(5);
  }

  private void assertExecutionMatchesSimulation(Consumer<AnalyticsPollersManager> startScheduler, BackoffSimulation simulation,
                                                int simulationIterations) {
    AnalyticsPollersManager pollersManager = pollersManager(simulatedClientProvider(simulation));
    startScheduler.accept(pollersManager);
    range(0, simulationIterations).forEach(i -> {
      scheduledRunnable(i).run();
      assertThat(iteration(i), executorLogger.scheduledTasks().get(i),
                 is(new ScheduledTask(null, simulation.delay(i), 0, MILLISECONDS)));
    });
  }

  private AnalyticsPollersManager pollersManager(ApiPlatformClient client) {
    AnalyticsConfiguration analyticsConfiguration = new AnalyticsConfiguration();
    AnalyticsPollersManager pollersManager =
        new AnalyticsPollersManager(analyticsConfiguration, mockCaches(), client, backoffSchedulerFactory());
    return withNoDispersion(pollersManager);
  }

  private AnalyticsEventCaches mockCaches() {
    AnalyticsEventCaches caches = mock(AnalyticsEventCaches.class, RETURNS_DEEP_STUBS);
    when(caches.getRegularEventsCache().isEmpty()).thenReturn(false);
    when(caches.getViolationsEventsCache().isEmpty()).thenReturn(false);
    when(caches.getRegularEventsCache().poll()).thenReturn(null);
    when(caches.getViolationsEventsCache().poll()).thenReturn(null);
    return caches;
  }

  private AnalyticsPollersManager withNoDispersion(AnalyticsPollersManager pollersManager) {
    BackoffConfiguration configuration = new Inspector(pollersManager).read("backoffConfiguration");
    overrideVariable("backoffConfiguration").in(pollersManager).with(withNoDispersion(configuration));
    return pollersManager;
  }

  @Override
  protected RangeDispersant expectedBackoffDispersion() {
    return new RangeDispersant(DISPERSION, 1);
  }

  @Override
  protected RangeDispersant expectedBackonDispersion() {
    return expectedBackoffDispersion();
  }

  private BackoffSchedulerFactory backoffSchedulerFactory() {
    return new FixedExecutorBackoffSchedulerFactory(new ObservableScheduledExecutorService(executorLogger));
  }

  private ApiPlatformClient simulatedClientProvider(BackoffSimulation backoffSimulation) {
    ApiPlatformClient client = mock(ApiPlatformClient.class, RETURNS_DEEP_STUBS);
    when(client.isConnected()).thenReturn(true);
    try {
      when(client.postHttpEvents(any())).thenAnswer(new Answer<Integer>() {

        int iteration = 1;

        @Override
        public Integer answer(InvocationOnMock invocation) {
          return backoffSimulation.statusCode(iteration++, 200);
        }
      });
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
    return client;
  }
}
