/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics.servicemesh;

import static com.mulesoft.mule.runtime.gw.analytics.servicemesh.ServiceMeshHeaderInjector.SEPARATOR;
import static com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolationOutcome.VIOLATION;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.util.MultiMap;

import com.mulesoft.mule.runtime.gw.analytics.model.AnalyticsHttpEventBuilder;
import com.mulesoft.mule.runtime.gw.analytics.model.HttpResponseAttributes;
import com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolation;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;

import java.util.Base64;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class ServiceMeshHeaderInjectorTestCase {

  public static final long API_ID = 25835985L;
  public static final int LEGACY_API_ID = 8908211;
  public static final String ORG_ID = "4d7005b1-8d68-4008-b6a5-f957be87bebe";
  public static final String CLIENT_ID = "3e2c539fbcf7461293a92976dc810e60";
  private static final String SM_HEADER = new AnalyticsConfiguration().getServiceMeshHeader();
  private ServiceMeshHeaderInjector injector = new ServiceMeshHeaderInjector();

  @Mock
  private Api api;

  @Mock
  private TrackingInfo trackingInfo;

  @Mock
  private ApiKey apiKey;

  private AnalyticsHttpEventBuilder builder;

  @Before
  public void setUp() {
    when(api.getTrackingInfo()).thenReturn(trackingInfo);
    when(api.getKey()).thenReturn(apiKey);
    when(apiKey.id()).thenReturn(API_ID);
    when(trackingInfo.getLegacyApiIdentifier()).thenReturn(LEGACY_API_ID);
    when(trackingInfo.getOrganizationId()).thenReturn(ORG_ID);
    builder = new AnalyticsHttpEventBuilder(null, null, null, null);
    builder.withApi(api);
  }

  @Test
  public void injectHeaderNoClientId() {
    HttpResponseAttributes attributes = new HttpResponseAttributes(MultiMap.emptyMultiMap(), 200);
    injector.injectHeader(attributes, builder.build());

    String header = attributes.getHeaders().get(SM_HEADER);
    String decoded = new String(Base64.getUrlDecoder().decode(header.getBytes()));
    String[] parts = decoded.split("\\" + SEPARATOR);

    assertThat(parts.length, is(5));
    assertThat(parts[0], is(ORG_ID));
    assertThat(parts[1], is(String.valueOf(LEGACY_API_ID)));
    assertThat(parts[2], is(String.valueOf(API_ID)));
    assertThat(parts[3], is("null"));
    assertThat(parts[4], is("processed"));
  }

  @Test
  public void injectHeaderWithClientId() {
    builder.withClient(new Client.ClientBuilder().withId(CLIENT_ID).withSecret("x").build());

    HttpResponseAttributes attributes = new HttpResponseAttributes(MultiMap.emptyMultiMap(), 200);
    injector.injectHeader(attributes, builder.build());

    String header = attributes.getHeaders().get(SM_HEADER);
    String decoded = new String(Base64.getUrlDecoder().decode(header.getBytes()));
    String[] parts = decoded.split("\\" + SEPARATOR);

    assertThat(parts.length, is(5));
    assertThat(parts[0], is(ORG_ID));
    assertThat(parts[1], is(String.valueOf(LEGACY_API_ID)));
    assertThat(parts[2], is(String.valueOf(API_ID)));
    assertThat(parts[3], is(CLIENT_ID));
    assertThat(parts[4], is("processed"));
  }

  @Test
  public void injectHeaderWithViolation() {
    PolicyViolation policyViolation = PolicyViolation.builder().withPolicyId("306664")
        .withPolicyName("basic-client-id-enforcement").withCategory("authentication").withOutcome(VIOLATION).build();
    builder.withPolicyViolation(policyViolation);

    HttpResponseAttributes attributes = new HttpResponseAttributes(MultiMap.emptyMultiMap(), 200);
    injector.injectHeader(attributes, builder.build());

    String header = attributes.getHeaders().get(SM_HEADER);
    String decoded = new String(Base64.getUrlDecoder().decode(header.getBytes()));
    String[] parts = decoded.split("\\" + SEPARATOR);

    assertThat(parts.length, is(9));
    assertThat(parts[0], is(ORG_ID));
    assertThat(parts[1], is(String.valueOf(LEGACY_API_ID)));
    assertThat(parts[2], is(String.valueOf(API_ID)));
    assertThat(parts[3], is("null"));
    assertThat(parts[4], is("blocked"));
    assertThat(parts[5], is("306664"));
    assertThat(parts[6], is("basic-client-id-enforcement"));
    assertThat(parts[7], is("authentication"));
    assertThat(parts[8], is("violation"));
  }
}
