/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.notification.EnrichedServerNotification;
import org.mule.runtime.api.util.MultiMap;

import com.mulesoft.mule.runtime.gw.analytics.extractor.ClientIpExtractor;
import com.mulesoft.mule.runtime.gw.analytics.extractor.DefaultClientIpExtractor;
import com.mulesoft.mule.runtime.gw.analytics.extractor.EnrichedServerNotificationExtractor;
import com.mulesoft.mule.runtime.gw.analytics.extractor.NoopClientIpExtractor;
import com.mulesoft.mule.runtime.gw.analytics.feedbackloop.EdgeRequestDetector;
import com.mulesoft.mule.runtime.gw.analytics.feedbackloop.FeedbackLoopDDOSHeaderInjector;
import com.mulesoft.mule.runtime.gw.analytics.model.AnalyticsEventBuilder;
import com.mulesoft.mule.runtime.gw.analytics.model.AnalyticsHttpEventBuilder;
import com.mulesoft.mule.runtime.gw.analytics.model.HttpRequestAttributes;
import com.mulesoft.mule.runtime.gw.analytics.model.HttpRequestAttributesFactory;
import com.mulesoft.mule.runtime.gw.analytics.model.HttpResponseAttributesFactory;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.service.EventsCollectorService;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.hdp.config.HighDensityProxyConfiguration;
import com.mulesoft.mule.runtime.gw.metrics.GatewayMetricsAdapter;
import com.mulesoft.mule.runtime.gw.model.Api;

import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles the creation and dispatch of the analytics events generated by the execution of an API. One per application is created,
 * even if an application contains more than one API.
 */
public class AnalyticsEventHandler {

  private static final Logger LOGGER = LoggerFactory.getLogger(AnalyticsEventHandler.class);

  private final String muleAppName;
  private final ApiService apiService;
  private final AnalyticsEventDispatcher eventDispatcher;
  private final EnrichedServerNotificationExtractor extractor;
  private final Map<String, AnalyticsEventBuilder> eventBuilders = new ConcurrentHashMap<>();
  private final EventsCollectorService eventsCollectorService;

  private final HttpRequestAttributesFactory httpRequestAttributesFactory = new HttpRequestAttributesFactory();
  private final HttpResponseAttributesFactory httpResponseAttributesFactory = new HttpResponseAttributesFactory();
  private final EdgeRequestDetector edgeRequestDetector = new EdgeRequestDetector();
  private final ClientIpExtractor clientIpExtractor;
  private Optional<GatewayMetricsAdapter> metricsCollector;

  private final FeedbackLoopDDOSHeaderInjector headerInjector = new FeedbackLoopDDOSHeaderInjector();

  private final HighDensityProxyConfiguration hdpConfiguration = new HighDensityProxyConfiguration();

  public AnalyticsEventHandler(String muleAppName, ApiService apiService, EnrichedServerNotificationExtractor extractor,
                               AnalyticsEventDispatcher eventDispatcher, EventsCollectorService eventsCollectorService,
                               Optional<GatewayMetricsAdapter> metricsCollector) {
    this.muleAppName = muleAppName;
    this.apiService = apiService;
    this.extractor = extractor;
    this.eventDispatcher = eventDispatcher;
    this.eventsCollectorService = eventsCollectorService;
    this.clientIpExtractor =
        new AnalyticsConfiguration().includeClientIp() ? new DefaultClientIpExtractor() : new NoopClientIpExtractor();
    this.metricsCollector = metricsCollector;
  }

  public void startEvent(EnrichedServerNotification notification) {
    if (!eventBuilders.containsKey(extractor.eventId(notification))) {
      String flow = extractor.flow(notification);
      Optional<Api> api = findApi(flow, notification);

      if (api.isPresent() && api.get().getTrackingInfo().isTracked()) {
        LOGGER.debug("Processing Message Received Notification for '{}'", api.get().getKey());

        OptionalLong payloadLength = notification.getEvent().getMessage().getPayload().getByteLength();

        AnalyticsEventBuilder eventBuilder =
            new AnalyticsHttpEventBuilder(edgeRequestDetector, httpRequestAttributesFactory, httpResponseAttributesFactory,
                                          clientIpExtractor)
                                              .withMuleAppName(muleAppName)
                                              .withFlowName(flow)
                                              .withId(extractor.eventId(notification))
                                              .withApi(api.get())
                                              .withReceivedTimestamp(notification.getTimestamp())
                                              .withRequest(payloadLength,
                                                           notification.getEvent().getMessage().getAttributes());

        eventBuilders.put(key(notification, flow), eventBuilder);
      }
    }
  }

  private Optional<Api> findApi(String flow, EnrichedServerNotification notification) {
    return apiService.find(muleAppName, flow, () -> {
      TypedValue<Object> attributes = notification.getEvent().getMessage().getAttributes();
      Optional<MultiMap<String, String>> headers =
          httpRequestAttributesFactory.from(attributes).map(HttpRequestAttributes::getHeaders);
      String service = null;
      String serviceHeader = hdpConfiguration.getHdpServiceHeader();
      if (headers.isPresent() && headers.get().containsKey(serviceHeader)) {
        service = headers.get().get(serviceHeader);
      }
      return service;
    });
  }

  public void finishEvent(EnrichedServerNotification notification) {

    AnalyticsEventBuilder eventBuilder = eventBuilders.remove(key(notification, extractor.flow(notification)));

    if (eventBuilder != null) {
      LOGGER.debug("Processing Message Response Notification for '{}'", eventBuilder.getApiDescription());

      OptionalLong payloadLength = notification.getEvent().getMessage().getPayload().getByteLength();
      TypedValue<Object> attributes = notification.getEvent().getMessage().getAttributes();

      eventBuilder.withResponse(payloadLength, attributes, notification.getAction())
          .withRepliedTimestamp(notification.getTimestamp());

      extractor.client(notification).ifPresent(eventBuilder::withClient);

      eventsCollectorService.removePolicyViolation(extractor.eventId(notification))
          .ifPresent(policyViolation -> {
            eventBuilder.withPolicyViolation(policyViolation);
            if (eventBuilder.isEdgeRequest()) {
              headerInjector.injectHeader(attributes.getValue(), policyViolation);
            }
          });

      dispatchEvent(eventBuilder.build(), attributes.getValue());
    }
  }

  void dispatchEvent(AnalyticsHttpEvent event, Object attributes) {
    eventDispatcher.dispatch(event);
    metricsCollector.ifPresent(gatewayMetricsAdapter -> gatewayMetricsAdapter.requestReceived(event));
  }

  private String key(EnrichedServerNotification notification, String flow) {
    return extractor.eventId(notification) + "-" + flow;
  }
}
