/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics;

import com.mulesoft.mule.runtime.gw.analytics.cache.AnalyticsEventCacheManager;
import com.mulesoft.mule.runtime.gw.analytics.extractor.EnrichedServerNotificationExtractor;
import com.mulesoft.mule.runtime.gw.api.service.EventsCollectorService;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.metrics.GatewayMetricsAdapter;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * In charge of creating, storing and removing analytics events handlers for each application
 */
public class AnalyticsEventHandlerManager {

  private final ApiService apiService;
  private final AnalyticsEventCacheManager cacheManager;
  private final Boolean serviceMesh;
  private final Map<String, AnalyticsEventHandler> handlers;
  private EventsCollectorService eventsCollectorService;
  private Optional<GatewayMetricsAdapter> metricsCollector;

  public AnalyticsEventHandlerManager(ApiService apiService, AnalyticsEventCacheManager cacheManager,
                                      EventsCollectorService eventsCollectorService, Boolean serviceMesh,
                                      Optional<GatewayMetricsAdapter> metricsCollector) {
    this.apiService = apiService;
    this.cacheManager = cacheManager;
    this.serviceMesh = serviceMesh;
    this.handlers = new HashMap<>();
    this.eventsCollectorService = eventsCollectorService;
    this.metricsCollector = metricsCollector;
  }

  /**
   * Creates a new {@link AnalyticsEventHandler} for the specified application
   * 
   * @param appName the application name
   * @return the created handler
   */
  public AnalyticsEventHandler create(String appName) {
    AnalyticsEventHandler handler;
    EnrichedServerNotificationExtractor extractor = new EnrichedServerNotificationExtractor();
    if (serviceMesh) {
      handler = new AnalyticsServiceMeshEventHandler(appName, apiService, extractor, eventsCollectorService);
    } else {
      handler = new AnalyticsEventHandler(appName, apiService, extractor, new AnalyticsEventDispatcher(cacheManager),
                                          eventsCollectorService, metricsCollector);
    }
    handlers.put(appName, handler);
    return handler;
  }

  /**
   * @param appName the application's name
   * @return the {@link AnalyticsEventHandler} of the specified application
   */
  public AnalyticsEventHandler get(String appName) {
    return handlers.get(appName);
  }

  /**
   * Removes the handler of the specified application
   * 
   * @param appName the application's name
   */
  public void remove(String appName) {
    handlers.remove(appName);
  }
}
