/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.analytics;

import static com.mulesoft.mule.runtime.gw.api.time.period.Period.seconds;
import static com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration.configuration;

import java.util.concurrent.Executors;

import org.mule.runtime.core.api.util.concurrent.NamedThreadFactory;

import com.mulesoft.mule.runtime.gw.analytics.backoff.AnalytictsBackoffConfigurationSupplier;
import com.mulesoft.mule.runtime.gw.analytics.cache.AnalyticsEventCaches;
import com.mulesoft.mule.runtime.gw.analytics.runnable.AnalyticsEventSenderRunnable;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.BackoffScheduler;
import com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.anypoint.backoff.session.BackoffBarrier;
import com.mulesoft.anypoint.backoff.session.PlatformBackoffBarrier;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;

public class AnalyticsPollersManager {

  private BackoffSchedulerFactory schedulerFactory;
  private BackoffScheduler analyticsScheduler;
  private BackoffScheduler policyViolationsScheduler;
  private AnalyticsConfiguration configuration;
  private AnalyticsEventCaches cloudCaches;
  private ApiPlatformClient restClient;
  private BackoffConfiguration backoffConfiguration;

  public AnalyticsPollersManager(AnalyticsConfiguration configuration, AnalyticsEventCaches cloudCaches,
                                 ApiPlatformClient apiPlatformClient, BackoffSchedulerFactory schedulerFactory) {
    this.configuration = configuration;
    this.cloudCaches = cloudCaches;
    this.restClient = apiPlatformClient;
    this.schedulerFactory = schedulerFactory;
    this.backoffConfiguration = new AnalytictsBackoffConfigurationSupplier().get();
  }

  public void scheduleAnalyticsRunnable() {
    if (analyticsScheduler == null) {

      analyticsScheduler =
          schedulerFactory.create(Executors.newScheduledThreadPool(1, new NamedThreadFactory("agw-analytics")));

      // TODO: AGW-1285: Use Mule's Scheduler Service
      AnalyticsEventSenderRunnable runnable = new AnalyticsEventSenderRunnable(cloudCaches.getRegularEventsCache(),
                                                                               configuration.getAnalyticsEventsBatchSize(),
                                                                               restClient, backoffConfiguration,
                                                                               backoffBarrier());
      analyticsScheduler
          .scheduleWithFixedDelay(runnable, analyticsSchedulerConfiguration());
    }
  }

  public void schedulePolicyViolationsRunnable() {
    if (policyViolationsScheduler == null) {
      policyViolationsScheduler =
          schedulerFactory.create(Executors.newScheduledThreadPool(1, new NamedThreadFactory("agw-policy-violations")));

      AnalyticsEventSenderRunnable runnable =
          new AnalyticsEventSenderRunnable(cloudCaches.getViolationsEventsCache(),
                                           configuration.getAnalyticsEventsBatchSize(), restClient, backoffConfiguration,
                                           backoffBarrier());
      policyViolationsScheduler
          .scheduleWithFixedDelay(runnable, policyViolationsSchedulerConfiguration());
    }
  }

  private BackoffBarrier backoffBarrier() {
    return new PlatformBackoffBarrier();
  }

  public SchedulingConfiguration analyticsSchedulerConfiguration() {
    int pushFreq = configuration.getPushFrequency();
    return configuration(seconds(pushFreq), seconds(pushFreq));
  }

  public SchedulingConfiguration policyViolationsSchedulerConfiguration() {
    int pushFreq = configuration.getPolicyViolationsPushFrequency();
    return configuration(seconds(pushFreq + 2), seconds(pushFreq));
  }

  public void shutdown() {
    analyticsScheduler.dispose();
    policyViolationsScheduler.dispose();
  }

}
