/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment;

import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.deployment.DeploymentTarget.getDeploymentTarget;
import static com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypterFactory.createDefaultRuntimeEncrypter;
import static java.util.Optional.empty;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.config.custom.CustomizationService;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.healthcheck.HealthCheckValidator;
import org.mule.runtime.api.healthcheck.ReadyStatus;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.container.api.MuleCoreExtensionDependency;
import org.mule.runtime.core.internal.util.OneTimeWarning;
import org.mule.runtime.module.deployment.api.DeploymentListener;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.deployment.api.StartupListener;

import com.mulesoft.mule.runtime.gw.api.agent.GatewayCoreExtension;
import com.mulesoft.mule.runtime.gw.api.agent.HealthCheck;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration;
import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.mule.runtime.gw.autodiscovery.ApiDiscovery;
import com.mulesoft.mule.runtime.gw.backoff.configuration.BackoffConfigurationSupplier;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientProvider;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.deployment.contracts.ContractSnapshots;
import com.mulesoft.mule.runtime.gw.deployment.replication.ApiConfigurationCache;
import com.mulesoft.mule.runtime.gw.deployment.replication.DistributedApiConfigurationCache;
import com.mulesoft.mule.runtime.gw.deployment.replication.StandaloneApiConfigurationCache;
import com.mulesoft.mule.runtime.gw.deployment.service.CoreServicesClientsRepository;
import com.mulesoft.mule.runtime.gw.deployment.service.DefaultApiService;
import com.mulesoft.mule.runtime.gw.hdp.listener.HdpDeploymentListener;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.deployment.tracking.DefaultApiTrackingService;
import com.mulesoft.mule.runtime.gw.extension.GatewayEntitledCoreExtension;
import com.mulesoft.mule.runtime.gw.logging.LoggingClassLoaderSelector;
import com.mulesoft.mule.runtime.gw.metrics.GatewayMetricsAdapter;
import com.mulesoft.mule.runtime.gw.metrics.GatewayMetricsFactory;
import com.mulesoft.mule.runtime.gw.model.contracts.ClientFactory;
import com.mulesoft.mule.runtime.gw.model.contracts.DefaultClientFactory;
import com.mulesoft.mule.runtime.gw.model.contracts.HashedClientFactory;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.encryption.DefaultPolicyConfigurationEncrypter;
import com.mulesoft.mule.runtime.gw.policies.factory.DefaultPolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.factory.EncryptedPolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.DefaultHealthCheck;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.GateKeeperSupplier;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.HdpApisHealthCheckListener;
import com.mulesoft.mule.runtime.gw.policies.notification.PolicyNotificationListenerSuppliers;
import com.mulesoft.mule.runtime.gw.policies.offline.OfflinePolicyWatcher;
import com.mulesoft.mule.runtime.gw.policies.service.DefaultPolicyDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.service.DefaultPolicyDeploymentTracker;
import com.mulesoft.mule.runtime.gw.policies.service.DefaultPolicySetDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore;
import com.mulesoft.mule.runtime.gw.policies.store.EncryptedPropertiesSerializer;
import com.mulesoft.mule.runtime.gw.policies.template.provider.FileSystemPolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.resolver.HandlebarsPolicyTemplateResolver;
import com.mulesoft.mule.runtime.gw.retry.BackoffRunnableRetrierFactory;
import com.mulesoft.mule.runtime.module.cluster.api.ClusterCoreExtension;
import com.mulesoft.mule.runtime.module.cluster.api.notification.PrimaryClusterNodeListener;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterCoreExtension;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterManager;

import java.util.HashMap;
import java.util.Optional;
import java.util.concurrent.locks.ReentrantLock;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApiDeploymentCoreExtension extends GatewayEntitledCoreExtension implements
    GatewayCoreExtension, PrimaryClusterNodeListener, StartupListener {

  private static final Logger LOGGER = LoggerFactory.getLogger(ApiDeploymentCoreExtension.class);
  private static final String AGW_HEALTHCHECK_HOLDER = "agw-healthcheck-holder";

  private final long startTime = System.currentTimeMillis();

  private final GatewayConfiguration configuration;
  private final ApiPlatformClientProvider clientProvider;
  private final BackoffConfigurationSupplier backoffConfigurationSupplier;
  private final OneTimeWarning missingClientCredentialErrorWarn;
  private final BackoffRunnableRetrierFactory backoffRunnableRetrierFactory;

  private ApiDiscovery apiDiscovery;

  @Inject
  private ContractService contractService;

  @Inject
  private DeploymentService deploymentService;

  private ClusterCoreExtension clusterCoreExtension;
  private OfflinePolicyWatcher offlinePolicyWatcher;
  private ApiPlatformClient restClient;
  private PolicyNotificationListenerSuppliers notificationListenerSuppliers;
  private ApiService apiService;
  private HdpDeploymentListener hdpDeploymentListener;
  private ApiTrackingService apiTrackingService;
  private ApiConfigurationCache apiConfigurationCache;
  private ContractSnapshots contractSnapshots;
  private PlatformInteractionManager platformInteractionManager;
  private ApiDeploymentListener offlineModeDeploymentListener;
  private PolicySetDeploymentService policySetDeploymentService;
  private HealthCheck healthCheck;
  private boolean extensionInitialized;
  private boolean restClientInitialized;
  private ClientFactory clientFactory;
  private Optional<GatewayMetricsAdapter> metricsCollector = empty();

  public ApiDeploymentCoreExtension() {
    this.configuration = new GatewayConfiguration();
    this.backoffConfigurationSupplier = new BackoffConfigurationSupplier();
    this.backoffRunnableRetrierFactory = new BackoffRunnableRetrierFactory(this.configuration);
    this.clientProvider = new ApiPlatformClientProvider(backoffRunnableRetrierFactory.platformConnectionRetrier());
    this.missingClientCredentialErrorWarn =
        new OneTimeWarning(LOGGER, "Client ID or Client Secret were not provided. API Platform client is DISABLED.");
  }

  @Override
  public void initialiseCoreExtension() {
    this.apiDiscovery = new ApiDiscovery();
    this.notificationListenerSuppliers = new PolicyNotificationListenerSuppliers();
    this.apiService = new DefaultApiService(deploymentService);
    this.hdpDeploymentListener = new HdpDeploymentListener(apiService, deploymentService);
  }

  @Override
  public String getName() {
    return "API Gateway Extension";
  }

  public ApiPlatformClient apiPlatformClient() {
    return clientProvider.getClient();
  }

  public synchronized void finishLazyInitialization() {
    if (!extensionInitialized) {
      finishCoreExtensionInitialization();
      startCoreExtension();
      extensionInitialized = true;
    }
    if (!restClientInitialized) {
      if (onlineMode()) {
        initialiseRestClient();

      } else {
        missingClientCredentialErrorWarn.warn();
      }
    }
  }

  public void finishCoreExtensionInitialization() {
    HazelcastClusterManager hazelcastManager = ((HazelcastClusterCoreExtension) clusterCoreExtension).getHazelcastManager();

    if (hazelcastManager != null) {
      hazelcastManager.registerPrimaryNodeListener(this);
    }

    LOGGER.info("Starting {} in {} mode", getName(), hazelcastManager != null ? "CLUSTERED" : "STANDALONE");

    if (configuration.platformClient().isOnPrem()) {
      LOGGER.info("Running in On Prem mode.");
    }

    if (configuration.securityConfiguration().isEncryptionEnabled()) {
      LOGGER.debug("An encryption key is present. Policies and API contracts will be encrypted");
    } else {
      LOGGER.debug("No encryption key provided. Policies and API contracts won't be encrypted");
    }

    this.restClient = clientProvider.getClient();
    this.contractService.contractSupplier(this.apiService);
    this.contractService.contractPrefetch(this.apiService);

    DefaultPolicyDeploymentTracker policyDeploymentTracker = new DefaultPolicyDeploymentTracker();
    DefaultPolicyStore policyStore = new DefaultPolicyStore(new EncryptedPropertiesSerializer());
    PolicyDeploymentService policyDeploymentService =
        new DefaultPolicyDeploymentService(apiService, notificationListenerSuppliers,
                                           policyDeploymentTracker,
                                           policyStore);
    this.policySetDeploymentService =
        new DefaultPolicySetDeploymentService(backoffRunnableRetrierFactory.policySetDeploymentServiceRetrier(),
                                              policyDeploymentService, policyDeploymentTracker,
                                              policyStore, getPolicyFactory(), apiService);
    this.offlinePolicyWatcher = new OfflinePolicyWatcher(policyDeploymentService, getPolicyFactory());
    this.contractSnapshots = new ContractSnapshots(apiService, new ReentrantLock(), getClientFactory());

    this.apiConfigurationCache = hazelcastManager != null
        ? new DistributedApiConfigurationCache(apiService, policySetDeploymentService, contractSnapshots,
                                               hazelcastManager)
        : new StandaloneApiConfigurationCache(policySetDeploymentService);

    this.offlineModeDeploymentListener =
        new OfflineModeApiDeploymentListener(apiConfigurationCache, policySetDeploymentService);
    this.apiTrackingService =
        new DefaultApiTrackingService(apiService, policySetDeploymentService, apiConfigurationCache, contractSnapshots,
                                      contractService, configuration.onApiDeletedConfiguration());

    HdpApisHealthCheckListener hdpApisHealthCheckListener = new HdpApisHealthCheckListener(apiService);

    policySetDeploymentService.addPolicyDeploymentListener(hdpApisHealthCheckListener);
    apiService.addApiContractsListener(hdpApisHealthCheckListener);
    apiService.addDeploymentListener(hdpApisHealthCheckListener);

    deploymentService.addDeploymentListener((DeploymentListener) apiService);
    deploymentService.addDeploymentListener(hdpDeploymentListener);

    apiService.addDeploymentListener(policySetDeploymentService);
    apiService.addDeploymentListener(offlineModeDeploymentListener);

    this.metricsCollector = new GatewayMetricsFactory()
        .from(restClient, configuration, deploymentService, apiService, policyDeploymentTracker);
    collectGatewayMetrics();

    offlinePolicyWatcher.initialise();
    apiConfigurationCache.initialise(apiTrackingService);

    this.healthCheck = new DefaultHealthCheck(apiService);
  }

  public void startCoreExtension() {
    deploymentService.addStartupListener(this);
    try {
      offlinePolicyWatcher.start();
    } catch (Exception e) {
      LOGGER.error("An unexpected exception was raised when loading in Gateway Pollers Core Extension. {}", errorMessage(e));
    }

    new GateKeeperSupplier(configuration.gateKeeper(), apiService).get().ifPresent(gateKeeper -> {
      policySetDeploymentService.addPolicyDeploymentListener(gateKeeper);
      apiService.addApiContractsListener(gateKeeper);
      apiService.addDeploymentListener(gateKeeper);
    });
  }

  @Override
  public void onArtifactCreated(String artifactName, CustomizationService customizationService) {
    LoggingClassLoaderSelector.initialise(Thread.currentThread().getContextClassLoader());
    withContextClassLoader(containerClassLoader.getClassLoader(), () -> {
      if (extensionLoaded()) {
        customizationService.registerCustomServiceImpl("api-deployment-initialization", new ExtensionInitialisation(this));
        if (onlineMode()) {
          customizationService.registerCustomServiceImpl("clients-repository",
                                                         new CoreServicesClientsRepository(clientProvider.getClient()));
        }

        customizationService.registerCustomServiceImpl("hdp-apis-healthcheck", new HashMap<String, HealthCheckValidator>());
      }

      // Register health check validator holder no matter if we have gateway entitlements
      customizationService.registerCustomServiceImpl(AGW_HEALTHCHECK_HOLDER, new HealthCheckValidatorHolder(artifactName));
      LOGGER.debug("HealthCheckValidatorHolder registered for application " + artifactName);
    });
  }

  @Override
  public void start() {}

  @Override
  public void stop() throws MuleException {
    removeSecretsFromSystem();
    clientProvider.shutdown();
    if (offlinePolicyWatcher != null) {
      offlinePolicyWatcher.stop();
    }
  }

  @Override
  public void dispose() {
    metricsCollector.ifPresent(GatewayMetricsAdapter::dispose);
    if (platformInteractionManager != null) {
      platformInteractionManager.dispose();
    }
    if (offlinePolicyWatcher != null) {
      offlinePolicyWatcher.dispose();
    }
    backoffRunnableRetrierFactory.dispose();
    LoggingClassLoaderSelector.dispose();
  }

  @Override
  public void onNotification() {
    LOGGER.info("We have become the primary cluster node. Starting API Manager runnables");
    if (restClient.isConnected()) {
      platformInteractionManager.primaryNode();
    }
  }

  @Override
  public void onAfterStartup() {
    if (LOGGER.isDebugEnabled()) {
      long elapsedTime = System.currentTimeMillis() - startTime;
      LOGGER.debug("-= All applications started (took " + elapsedTime + "ms)");
    }
  }

  @MuleCoreExtensionDependency
  public void setClusterCoreExtension(ClusterCoreExtension clusterCoreExtension) {
    this.clusterCoreExtension = clusterCoreExtension;
  }

  public ApiService getApiService() {
    return apiService;
  }

  public PolicyNotificationListenerSuppliers getNotificationListenerSuppliers() {
    return notificationListenerSuppliers;
  }

  /**
   * Expose runtime healthCheck instance to be used by the Mule Agent
   */
  @Override
  public HealthCheck healthCheck() {
    return healthCheck;
  }

  public Optional<GatewayMetricsAdapter> metricsCollector() {
    return metricsCollector;
  }

  void initialiseRestClient() {
    // Execute with container class loader to log rest initialization messages in mule-ee log
    if (clientProvider.configureClient(configuration)) {
      platformInteractionManager = new PlatformInteractionManager(apiService, apiTrackingService,
                                                                  restClient,
                                                                  backoffConfigurationSupplier,
                                                                  isStandaloneOrPrimaryNode(),
                                                                  backoffRunnableRetrierFactory);
      clientProvider.addConnectionListener(platformInteractionManager);
      clientProvider.addConnectionListener(() -> apiService.removeDeploymentListener(offlineModeDeploymentListener));
      metricsCollector.ifPresent(clientProvider::addConnectionListener);
      clientProvider.connectClient();
      restClientInitialized = true;
    }
  }

  private void collectGatewayMetrics() {
    metricsCollector.ifPresent(gatewayMetricsAdapter -> gatewayMetricsAdapter
        .gatewayInformation(getProductVersion(), getDeploymentTarget(),
                            ((HazelcastClusterCoreExtension) clusterCoreExtension).getHazelcastManager() != null, configuration,
                            new AnalyticsConfiguration()));
  }

  private boolean isStandaloneOrPrimaryNode() {
    HazelcastClusterManager hazelcastManager = ((HazelcastClusterCoreExtension) clusterCoreExtension).getHazelcastManager();
    return hazelcastManager == null || hazelcastManager.isPrimaryPollingInstance();
  }

  private void removeSecretsFromSystem() {
    configuration.platformClient().clearClientSecret();
    configuration.securityConfiguration().clearEncryptionKey();
  }

  private PolicyFactory getPolicyFactory() {
    return configuration.securityConfiguration().isEncryptionEnabled()
        ? new EncryptedPolicyFactory(new HandlebarsPolicyTemplateResolver(),
                                     new FileSystemPolicyTemplateProvider(restClient),
                                     new DefaultPolicyConfigurationEncrypter(createDefaultRuntimeEncrypter(),
                                                                             configuration.securityConfiguration()
                                                                                 .isSensitiveOnlyEncryption()))
        : new DefaultPolicyFactory(new HandlebarsPolicyTemplateResolver(),
                                   new FileSystemPolicyTemplateProvider(restClient));
  }

  private ClientFactory getClientFactory() {
    GatewaySecurityConfiguration gatewaySecurityConfiguration = configuration.securityConfiguration();
    if (gatewaySecurityConfiguration.hashClients()) {
      String hashAlgorithm = gatewaySecurityConfiguration.hashAlgorithm();
      LOGGER.debug("API Gateway initializing with Client Hashing algorithm: {}", hashAlgorithm);
      return new HashedClientFactory(hashAlgorithm);
    }
    return new DefaultClientFactory();
  }

  class ExtensionInitialisation implements Initialisable {

    private ApiDeploymentCoreExtension apiDeploymentCoreExtension;

    @Inject
    private Registry registry;

    public ExtensionInitialisation(ApiDeploymentCoreExtension apiDeploymentCoreExtension) {
      this.apiDeploymentCoreExtension = apiDeploymentCoreExtension;
    }

    @Override
    public void initialise() {
      withContextClassLoader(containerClassLoader.getClassLoader(), () -> {
        if (onlineMode() || !apiDiscovery.apiKeys(registry).isEmpty()) {
          this.apiDeploymentCoreExtension.finishLazyInitialization();
        }
        if (!apiDiscovery.apiKeys(registry).isEmpty()) {
          registry.lookupByName(AGW_HEALTHCHECK_HOLDER).ifPresent(h -> ((HealthCheckValidatorHolder) h).set(healthCheck));
        }
      });
    }
  }

  private static class HealthCheckValidatorHolder implements HealthCheckValidator {

    private String artifactName;
    private HealthCheckValidator validator;

    public HealthCheckValidatorHolder(String artifactName) {
      this.artifactName = artifactName;
    }

    void set(HealthCheck healthCheck) {
      LOGGER.debug("HealthCheckValidatorHolder initialized for application " + artifactName);
      this.validator = healthCheck.getValidator(artifactName);
    }

    @Override
    public ReadyStatus ready() {
      if (validator != null) {
        return validator.ready();
      }
      return () -> true;
    }

  }

}
