/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment;

import static com.mulesoft.mule.runtime.gw.api.time.period.Period.seconds;
import static com.mulesoft.mule.runtime.gw.backoff.scheduler.configuration.SchedulingConfiguration.configuration;
import static java.util.concurrent.Executors.newScheduledThreadPool;

import org.mule.runtime.core.api.util.concurrent.NamedThreadFactory;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.backoff.configuration.BackoffConfiguration;
import com.mulesoft.mule.runtime.gw.backoff.configuration.BackoffConfigurationSupplier;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.BackoffScheduler;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.configuration.SchedulingConfiguration;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.runnable.BackoffRunnable;
import com.mulesoft.mule.runtime.gw.backoff.session.BackoffBarrier;
import com.mulesoft.mule.runtime.gw.backoff.session.PlatformBackoffBarrier;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients.PlatformClientsRetriever;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ApiKeepAliveRunnable;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ApisRunnable;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ClientsRunnable;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GatewayPollersManager {

  private static final Logger LOGGER = LoggerFactory.getLogger(GatewayPollersManager.class);
  private final GatewayConfiguration configuration;
  private final ApiService apiService;
  private final ApiTrackingService apiTrackingService;
  private final ApiPlatformSessionFactory platformSessionFactory;
  private final BackoffSchedulerFactory schedulerFactory;
  private final BackoffConfigurationSupplier backoffConfigurationSupplier;
  private BackoffScheduler schedulerKeepAlive;
  private BackoffScheduler schedulerApis;
  private BackoffScheduler schedulerClients;
  private ApiKeepAliveRunnable keepAliveRunnable;
  private ApisRunnable apisRunnable;
  private ClientsRunnable clientsRunnable;


  public GatewayPollersManager(GatewayConfiguration configuration,
                               ApiService apiService,
                               ApiTrackingService apiTrackingService,
                               ApiPlatformSessionFactory platformSessionFactory,
                               BackoffSchedulerFactory schedulerFactory,
                               BackoffConfigurationSupplier backoffConfigurationSupplier) {
    this.configuration = configuration;
    this.apiService = apiService;
    this.apiTrackingService = apiTrackingService;
    this.platformSessionFactory = platformSessionFactory;
    this.schedulerFactory = schedulerFactory;
    this.backoffConfigurationSupplier = backoffConfigurationSupplier;
  }

  public BackoffRunnable scheduleKeepAliveRunnable() {
    if (schedulerKeepAlive == null) {
      schedulerKeepAlive = schedulerFactory.create(newScheduledThreadPool(1,
                                                                          new NamedThreadFactory("agw-api-keep-alive")));
      keepAliveRunnable = new ApiKeepAliveRunnable(apiTrackingService,
                                                   platformSessionFactory,
                                                   backoffBarrier(),
                                                   backoffConfiguration());
      schedulerKeepAlive.scheduleWithFixedDelay(keepAliveRunnable, keepAliveConfiguration());

      LOGGER.debug("Keep Alive Runnable started successfully");
    }

    return keepAliveRunnable;
  }


  public BackoffRunnable scheduleApisPollerRunnable() {
    if (schedulerApis == null) {
      schedulerApis = schedulerFactory.create(newScheduledThreadPool(1,
                                                                     new NamedThreadFactory("agw-policy-polling")));

      apisRunnable = new ApisRunnable(apiTrackingService,
                                      platformSessionFactory,
                                      backoffBarrier(),
                                      backoffConfiguration());
      schedulerApis.scheduleWithFixedDelay(apisRunnable, apisConfiguration());

      LOGGER.debug("Policies Runnable started successfully");
    }

    return apisRunnable;
  }

  public BackoffRunnable scheduleClientsPollerRunnable() {
    if (schedulerClients == null) {
      schedulerClients = schedulerFactory.create(newScheduledThreadPool(1, new NamedThreadFactory("agw-contract-polling")));

      PlatformClientsRetriever retriever = new PlatformClientsRetriever(platformSessionFactory, apiTrackingService);
      apiService.addApiContractsListener(retriever);


      clientsRunnable = new ClientsRunnable(apiTrackingService,
                                            platformSessionFactory,
                                            backoffBarrier(),
                                            backoffConfiguration(),
                                            retriever);
      schedulerClients.scheduleWithFixedDelay(clientsRunnable, clientsConfiguration());

      LOGGER.debug("Contracts Runnable started successfully");
    }

    return clientsRunnable;
  }

  public void shutdown() {
    disposeScheduler(schedulerKeepAlive);
    disposeScheduler(schedulerApis);
    disposeScheduler(schedulerClients);
  }

  public SchedulingConfiguration keepAliveConfiguration() {
    return configuration(seconds(0), configuration.platformClient().getApiKeepAliveFrequency());
  }

  public SchedulingConfiguration apisConfiguration() {
    // delay policy polling start time with respect to the api keep alive thread in order to interleave requests
    return configuration(seconds(2), configuration.platformClient().getApisPollFrequency());
  }

  public SchedulingConfiguration clientsConfiguration() {
    // delay clients polling start time with respect to the api keep alive and policies thread in order to interleave requests
    return configuration(seconds(4), configuration.platformClient().getClientsPollFrequency());
  }

  private BackoffBarrier backoffBarrier() {
    return new PlatformBackoffBarrier();
  }

  private BackoffConfiguration backoffConfiguration() {
    return backoffConfigurationSupplier.forScheduling(configuration);
  }

  private void disposeScheduler(BackoffScheduler scheduler) {
    if (scheduler != null) {
      scheduler.dispose();
    }
  }

}
