/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment;

import org.mule.runtime.api.lifecycle.Disposable;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.backoff.configuration.BackoffConfigurationSupplier;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.factory.VariableExecutorBackoffSchedulerFactory;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientConnectionListener;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.client.session.factory.RestClientApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTracker;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.retry.BackoffRunnableRetrierFactory;

import org.slf4j.Logger;

public class PlatformInteractionManager implements ApiPlatformClientConnectionListener, Disposable {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(ApiDeploymentCoreExtension.class);

  private final ApiService apiService;
  private final ApiTrackingService apiTrackingService;
  private final ApiPlatformSessionFactory platformSessionFactory;
  private final GatewayConfiguration configuration;
  private final BackoffConfigurationSupplier backoffConfigurationSupplier;
  private final BackoffSchedulerFactory backoffSchedulerFactory;
  private final boolean standaloneOrPrimaryNode;
  private final BackoffRunnableRetrierFactory backoffRunnableRetrierFactory;

  private GatewayPollersManager pollersManager;
  private ApiTracker apiTracker;

  public PlatformInteractionManager(ApiService apiService,
                                    ApiTrackingService apiTrackingService,
                                    ApiPlatformClient restClient,
                                    BackoffConfigurationSupplier backoffConfigurationSupplier,
                                    boolean standaloneOrPrimaryNode,
                                    BackoffRunnableRetrierFactory backoffRunnableRetrierFactory) {
    this.apiService = apiService;
    this.apiTrackingService = apiTrackingService;
    this.configuration = new GatewayConfiguration();
    this.platformSessionFactory = new RestClientApiPlatformSessionFactory(restClient);
    this.backoffSchedulerFactory = new VariableExecutorBackoffSchedulerFactory();
    this.backoffConfigurationSupplier = backoffConfigurationSupplier;
    this.standaloneOrPrimaryNode = standaloneOrPrimaryNode;
    this.backoffRunnableRetrierFactory = backoffRunnableRetrierFactory;
  }

  @Override
  public void onClientConnected() {

    apiTracker = new ApiTracker(apiTrackingService, platformSessionFactory, backoffRunnableRetrierFactory.apiTrackerRetrier());

    /*
     * The listener must be set before the tracking.
     * Runtime could be deploying an app without the deployment listener being set.
     */
    apiService.addDeploymentListener(apiTracker);
    apiService.getApis().forEach(apiTracker::onApiDeploymentSuccess);
    pollersManager =
        new GatewayPollersManager(configuration, apiService, apiTrackingService, platformSessionFactory,
                                  backoffSchedulerFactory, backoffConfigurationSupplier);


    LOGGER.debug("Starting API Gateway runnables");

    if (standaloneOrPrimaryNode) {
      pollersManager.scheduleKeepAliveRunnable();
      pollersManager.scheduleApisPollerRunnable();
    }

    pollersManager.scheduleClientsPollerRunnable();
  }

  public void primaryNode() {
    pollersManager.scheduleKeepAliveRunnable();
    pollersManager.scheduleApisPollerRunnable();
  }

  @Override
  public void dispose() {
    if (pollersManager != null) {
      pollersManager.shutdown();
    }

    if (apiTracker != null) {
      apiTracker.dispose();
    }
  }
}
