/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.service;

import static java.util.Collections.synchronizedMap;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.healthcheck.HealthCheckValidator;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentListener;
import org.mule.runtime.module.deployment.api.DeploymentService;

import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.mule.runtime.gw.api.agent.HealthCheck;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.autodiscovery.ApiDiscovery;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.notification.ApiNotificationManager;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.contracts.ApiContractsFactory;
import com.mulesoft.mule.runtime.gw.model.hdp.ApiRegistry;
import com.mulesoft.mule.runtime.gw.notification.ApiContractsListener;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.DefaultHealthCheck;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.WeakHashMap;
import java.util.function.Predicate;
import java.util.function.Supplier;

import org.slf4j.Logger;

public class DefaultApiService implements ApiService, DeploymentListener {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(ApiService.class);

  private final ApiDiscovery apiDiscovery;
  private final DeploymentService deploymentService;
  private final ApiNotificationManager apiNotificationManager;
  private final Map<ApiKey, Api> apis = synchronizedMap(new WeakHashMap<>());
  private final HealthCheck healthCheck;
  private List<ApiContractsListener> apiContractsListeners = new LinkedList<>();

  public DefaultApiService(DeploymentService deploymentService) {
    this.deploymentService = deploymentService;
    this.apiDiscovery = new ApiDiscovery();
    this.apiNotificationManager = new ApiNotificationManager();
    this.healthCheck = new DefaultHealthCheck(this);
  }

  @Override
  public void onArtifactInitialised(String artifactName, Registry registry) {
    Application application = deploymentService.findApplication(artifactName);

    apiDiscovery.autoDiscoveryMetadatas(registry).forEach(apiMetadata -> {
      ApiKey apiKey = apiMetadata.getApiKey();

      synchronized (apis) {
        if (!apis.containsKey(apiKey)) {
          ApiImplementation implementation =
              new ApiImplementation(apiKey, application, apiMetadata.getFlow(), apiMetadata.isIgnoreBasePath());
          deployApi(artifactName, apiKey, implementation);
        } else {
          LOGGER.warn("API {} is already deployed on app {}. This API deployment won't be tracked.", apiMetadata.getApiKey(),
                      apis.get(apiKey).getImplementation().getArtifactName());
        }
      }
    });
  }

  private Api deployApi(String artifactName, ApiKey apiKey, ApiImplementation implementation) {
    LOGGER.debug("New API deployment {}, on app {}", apiKey, artifactName);
    Api api = new Api(apiKey, implementation, ApiContractsFactory.create(apiKey, apiContractsListeners));
    apis.put(implementation.getApiKey(), api);
    api.initialise();
    apiNotificationManager.notifyApiDeploymentStart(api);
    return api;
  }

  @Override
  public void addApiContractsListener(ApiContractsListener apiContractsListener) {
    apiContractsListeners.add(apiContractsListener);
  }

  @Override
  public void onDeploymentSuccess(String artifactName) {
    findApiByArtifactName(artifactName)
        .forEach(apiNotificationManager::notifyApiDeploymentSuccess);
  }

  @Override
  public void onDeploymentFailure(String artifactName, Throwable cause) {
    synchronized (apis) {
      findApiByArtifactName(artifactName).forEach(api -> apis.remove(api.getKey()));
    }
  }

  @Override
  public void onUndeploymentStart(String artifactName) {
    undeployApplicationApis(artifactName, api -> true);
  }

  private void undeployApplicationApis(String artifactName, Predicate<Api> filter) {
    synchronized (apis) {
      findApiByArtifactName(artifactName).stream().filter(filter).forEach(api -> {
        LOGGER.debug("API {} un-deployment started", api);
        apiNotificationManager.notifyApiUndeploymentStart(api.getImplementation());
        api.dispose();
        apis.remove(api.getKey());
      });
    }
  }

  @Override
  public void onRedeploymentStart(String artifactName) {
    synchronized (apis) {
      List<Api> redeployedApis = findApiByArtifactName(artifactName);

      redeployedApis.forEach(api -> {
        LOGGER.debug("API {} re-deployment started", api);
        apiNotificationManager.notifyApiRedeploymentStart(api.getImplementation());
        apis.remove(api.getKey());
      });
    }
  }

  @Override
  public boolean isDeployed(ApiKey apiKey) {
    return apis.containsKey(apiKey);
  }

  @Override
  public Optional<ApiImplementation> getImplementation(ApiKey apiKey) {
    synchronized (apis) {
      return apis.containsKey(apiKey) ? ofNullable(apis.get(apiKey).getImplementation()) : empty();
    }
  }

  @Override
  public Optional<Api> get(ApiKey apiKey) {
    return ofNullable(apis.get(apiKey));
  }

  @Override
  public Optional<Api> find(String appName, String flowName) {
    return apis.values().stream()
        .filter(internalApi -> flowName.equals(internalApi.getImplementation().getFlow().getName()) &&
            appName.equals(internalApi.getImplementation().getArtifactName()))
        .findFirst();
  }

  @Override
  public Optional<Api> find(String appName, String flowName, Supplier<String> serviceSupplier) {
    Optional<Api> api = find(appName, flowName);
    if (api.isPresent() && api.get().isOffline()) {
      return findHdpApi(appName, flowName, serviceSupplier.get());
    }
    return api;
  }

  private Optional<Api> findHdpApi(String appName, String flowName, String service) {
    if (service == null) {
      return empty();
    }
    return apis.values().stream()
        .filter(api -> flowName.equals(api.getImplementation().getFlow().getName()) &&
            appName.equals(api.getImplementation().getArtifactName()) && service.equals(api.getImplementation().getHdpService()
                .orElse(null)))
        .findFirst();
  }

  @Override
  public List<Api> getApis() {
    return new ArrayList<>(apis.values());
  }

  @Override
  public void addDeploymentListener(ApiDeploymentListener apiDeploymentListener) {
    this.apiNotificationManager.addApiDeploymentListener(apiDeploymentListener);
  }

  @Override
  public void removeDeploymentListener(ApiDeploymentListener apiDeploymentListener) {
    this.apiNotificationManager.removeApiDeploymentListener(apiDeploymentListener);
  }

  @Override
  public void updateHdpApis(ApiRegistry apiRegistry) {
    String appName = apiRegistry.getApplicationName();
    Application application = deploymentService.findApplication(appName);

    synchronized (apis) {
      // remove no longer tracked APIs
      undeployApplicationApis(appName, api -> !apiRegistry.containsApi(api) && !api.isOffline());

      // add new APIs
      apiRegistry.getApiRecords().forEach(t -> {
        if (!apis.containsKey(t.getApiKey())) {
          LOGGER.debug("New API deployment {}, on high density proxy {}. API blocked", t, appName);
          ApiImplementation implementation =
              new ApiImplementation(t.getApiKey(), application, getHdpFlow(application), t.getServiceName(), true);
          Api api = deployApi(appName, t.getApiKey(), implementation);

          application
              .getRegistry()
              .lookupByName("hdp-apis-healthcheck")
              .ifPresent(apisHealthCheck -> {
                ((Map<String, HealthCheckValidator>) apisHealthCheck)
                    .put(String.format("hdp-hc-%s-%s", appName, implementation.getApiKey().id().toString()),
                         healthCheck.getApiValidator(implementation.getApiKey().id()));
              });

          apiNotificationManager.notifyApiDeploymentSuccess(api);
        } else if (!appName.equals(apis.get(t.getApiKey()).getImplementation().getArtifactName())) {
          LOGGER.error("API {} is already deployed on app {}. This API deployment won't be tracked.", t.getApiKey(),
                       apis.get(t.getApiKey()).getImplementation().getArtifactName());
        }
      });

    }
  }

  private Flow getHdpFlow(Application application) {
    Optional<Flow> hdpFlow = application.getRegistry().lookupByType(Flow.class);
    return hdpFlow.orElseThrow(() -> new RuntimeException("HDP application has no flows"));
  }

  @Override
  public Optional<ApiContracts> getContracts(ApiKey key) {
    return get(key).map(Api::getContracts);
  }

  @Override
  public DefaultApiService contractsRequired(ApiKey key) {
    get(key).ifPresent(api -> apiContractsListeners.forEach(listener -> listener.onContractsRequired(api)));
    return this;
  }

  @Override
  public DefaultApiService noContractsRequired(ApiKey key) {
    get(key).ifPresent(api -> apiContractsListeners.forEach(listener -> listener.onNoContractsRequired(api)));
    return this;
  }

  private List<Api> findApiByArtifactName(String artifactName) {
    return getApis().stream()
        .filter(api -> artifactName.equals(api.getImplementation().getArtifactName()))
        .collect(toList());
  }

}
