/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.tracking;

import static com.mulesoft.mule.runtime.gw.model.NoTrackingInfo.trackingFailed;
import static com.mulesoft.mule.runtime.gw.model.NoTrackingInfo.untracked;
import static java.util.stream.Collectors.toList;

import com.mulesoft.mule.runtime.gw.api.config.OnApiDeletedConfiguration;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.contracts.ContractSnapshots;
import com.mulesoft.mule.runtime.gw.deployment.replication.ApiConfigurationCache;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;

import java.util.List;

import org.slf4j.Logger;

public class DefaultApiTrackingService implements ApiTrackingService {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(DefaultApiTrackingService.class);

  private final ApiService apiService;
  private final PolicySetDeploymentService policySetDeploymentService;
  private final ApiConfigurationCache apiConfigurationCache;
  private final ContractSnapshots contractSnapshots;
  private final ContractService contractService;
  private final OnApiDeletedConfiguration onApiDeletedConfiguration;

  public DefaultApiTrackingService(ApiService apiService,
                                   PolicySetDeploymentService policySetDeploymentService,
                                   ApiConfigurationCache apiConfigurationCache,
                                   ContractSnapshots contractSnapshots,
                                   ContractService contractService,
                                   OnApiDeletedConfiguration onApiDeletedConfiguration) {
    this.apiService = apiService;
    this.policySetDeploymentService = policySetDeploymentService;
    this.apiConfigurationCache = apiConfigurationCache;
    this.contractSnapshots = contractSnapshots;
    this.contractService = contractService;
    this.onApiDeletedConfiguration = onApiDeletedConfiguration;
  }

  @Override
  public void apiTracked(ApiKey apiKey, TrackingInfo trackingInfo, PolicySet policySet, List<Sla> slas) {
    apiService.get(apiKey).ifPresent(api -> {
      api.updateTrackingInfo(trackingInfo);
      api.getImplementation().gatekeeperStatus().tracked();
    });
    apiConfigurationCache.set(apiKey, policySet, slas);
    policySetDeploymentService.policiesForApi(apiKey, policySet);
    contractSnapshots.slas(apiKey, slas);
  }

  @Override
  public void apiUntracked(ApiKey apiKey) {
    if (onApiDeletedConfiguration.shouldBlock()) {
      LOGGER.debug("API {} is not tracked anymore, it will be blocked.", apiKey);
      apiConfigurationCache.remove(apiKey);
      policySetDeploymentService.removeAll(apiKey);
    } else {
      LOGGER.debug("API {} is not tracked anymore, current policies will be kept.", apiKey);
      // Deployed already stored policies if detected untracked during tracking phase
      apiConfigurationCache.getPolicies(apiKey)
          .ifPresent(policySet -> policySetDeploymentService.policiesForApi(apiKey, policySet));
    }

    apiService.get(apiKey).ifPresent(api -> {
      api.updateTrackingInfo(untracked());
      api.getImplementation().gatekeeperStatus().untracked();
    });
  }

  @Override
  public void apiTrackingFailed(ApiKey apiKey) {
    apiService.get(apiKey).ifPresent(api -> {
      if (!api.getTrackingInfo().isFailedTracking()) {
        LOGGER.debug("Marking API tracking for {} as failed", apiKey);
        api.updateTrackingInfo(trackingFailed());

        // Deployed already stored policies
        apiConfigurationCache.getPolicies(apiKey)
            .ifPresent(policySet -> policySetDeploymentService.policiesForApi(apiKey, policySet));
      }
    });
  }

  @Override
  public List<Api> getTrackedApis() {
    return apiService.getApis().stream().filter(this::isTracked).collect(toList());
  }

  @Override
  public List<Api> getFailedTrackingApis() {
    return apiService.getApis().stream().filter(api -> api.getTrackingInfo().isFailedTracking()).collect(toList());
  }

  @Override
  public List<Api> getTrackedApisRequiringContracts() {
    return apiService.getApis().stream()
        .filter(api -> isTracked(api) && requiresContracts(api))
        .collect(toList());
  }

  @Override
  public void clientsPolling(ApiKey apiKey, List<PlatformContractAdapter> contracts) {
    contractSnapshots.clients(apiKey, contracts);
  }

  private boolean isTracked(Api api) {
    return api.getTrackingInfo().isTracked();
  }

  private boolean requiresContracts(Api api) {
    return contractService.trackedApis().contains(api.getKey());
  }
}
