/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.replication;

import static com.mulesoft.mule.runtime.gw.model.PolicySet.PolicySetOrigin.FILE_SYSTEM;
import static com.mulesoft.mule.runtime.module.cluster.internal.config.HazelcastPropertiesConfig.HAZELCAST_GW_MAPS_PREFIX;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.concurrent.TimeUnit.SECONDS;

import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.contracts.ContractSnapshots;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.model.contracts.repository.ContractRepository;
import com.mulesoft.mule.runtime.gw.model.contracts.repository.MapDBContractRepository;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterManager;

import com.hazelcast.core.ILock;
import com.hazelcast.core.IMap;

import java.util.List;
import java.util.Map;
import java.util.Optional;

public class DistributedApiConfigurationCache implements ApiConfigurationCache {

  private static final String POLICIES_MAP_NAME =
      HAZELCAST_GW_MAPS_PREFIX.substring(0, HAZELCAST_GW_MAPS_PREFIX.length() - 1) + "policies";

  private final ApiService apiService;
  private final PolicySetDeploymentService policySetDeploymentService;
  private final ContractSnapshots contractSnapshots;
  private final HazelcastClusterManager hazelcastManager;
  // TODO: AGW-1634: Re think how to detect a first-time deployment
  private final ContractRepository contractRepository;

  private IMap<String, DistributedApiConfigurationEntry> entries;

  public DistributedApiConfigurationCache(ApiService apiService,
                                          PolicySetDeploymentService policySetDeploymentService,
                                          ContractSnapshots contractSnapshots, HazelcastClusterManager hazelcastManager) {
    this.apiService = apiService;
    this.policySetDeploymentService = policySetDeploymentService;
    this.contractSnapshots = contractSnapshots;
    this.hazelcastManager = hazelcastManager;
    this.entries = hazelcastManager.getHazelcastInstance().getMap(getPoliciesMapName());
    this.contractRepository = new MapDBContractRepository();
  }

  @Override
  public void set(ApiKey apiKey, PolicySet policySet, List<Sla> slas) {
    entries.put(getKey(apiKey), entry(policySet, slas));
  }

  @Override
  public void remove(ApiKey apiKey) {
    entries.remove(getKey(apiKey));
  }

  @Override
  public Optional<PolicySet> getPolicies(ApiKey apiKey) {
    if (entries.containsKey(getKey(apiKey))) {
      return safeGetEntry(apiKey).map(DistributedApiConfigurationEntry::getPolicySet);
    }

    return contractRepository.containsSla(apiKey) ? of(fileSystemSet(emptyList())) : empty();
  }

  @Override
  public void initialise(ApiTrackingService apiTrackingService) {
    Map<ApiKey, List<PolicyDefinition>> initialPolicies = policySetDeploymentService.storedOnlinePoliciesByApi();

    ILock lock = hazelcastManager.getHazelcastInstance().getLock(getPoliciesMapName());

    try {
      lock.lock(60, SECONDS);

      initialPolicies.forEach((apiKey, policyDefinitions) -> {
        if (hazelcastManager.isPrimaryPollingInstance()) {
          if (entries.containsKey(getKey(apiKey)) && getEntry(apiKey).getPolicySet().isFromPlatform()) {
            policySetDeploymentService
                .conciliatePolicies(apiKey, getEntry(apiKey).getPolicySet().getPolicyDefinitions());
          } else {
            entries.put(getKey(apiKey), entry(fileSystemSet(policyDefinitions), emptyList()));
          }
        } else {
          if (entries.containsKey(getKey(apiKey))) {
            policySetDeploymentService
                .conciliatePolicies(apiKey, getEntry(apiKey).getPolicySet().getPolicyDefinitions());
          } else {
            entries.put(getKey(apiKey), entry(fileSystemSet(policyDefinitions), emptyList()));
          }
        }
      });

      entries.addEntryListener(
                               new DistributedPoliciesMapEntryListener(apiService, apiTrackingService, policySetDeploymentService,
                                                                       contractSnapshots, hazelcastManager.isClientModeEnabled()),
                               true);
    } finally {
      lock.unlock();
    }
  }

  private String getKey(ApiKey apiKey) {
    return apiKey.id().toString();
  }

  private DistributedApiConfigurationEntry entry(PolicySet policySet, List<Sla> slas) {
    return new DistributedApiConfigurationEntry(policySet, slas);
  }

  private Optional<DistributedApiConfigurationEntry> safeGetEntry(ApiKey apiKey) {
    return ofNullable(entries.get(getKey(apiKey)));
  }

  private DistributedApiConfigurationEntry getEntry(ApiKey apiKey) {
    return entries.get(getKey(apiKey));
  }

  private PolicySet fileSystemSet(List<PolicyDefinition> policyDefinitions) {
    return new PolicySet(policyDefinitions, FILE_SYSTEM);
  }

  private String getPoliciesMapName() {
    return hazelcastManager.getClusterId() + "_" + POLICIES_MAP_NAME;
  }
}
