/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.replication;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.contracts.ContractSnapshots;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;

import com.hazelcast.core.EntryEvent;
import com.hazelcast.map.listener.EntryAddedListener;
import com.hazelcast.map.listener.EntryRemovedListener;
import com.hazelcast.map.listener.EntryUpdatedListener;

import org.slf4j.Logger;

public class DistributedPoliciesMapEntryListener
    implements EntryAddedListener<String, DistributedApiConfigurationEntry>,
    EntryUpdatedListener<String, DistributedApiConfigurationEntry>,
    EntryRemovedListener<String, DistributedApiConfigurationEntry> {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(DistributedPoliciesMapEntryListener.class);

  private final ApiService apiService;
  private final ApiTrackingService apiTrackingService;
  private final PolicySetDeploymentService policySetDeploymentService;
  private final ContractSnapshots contractSnapshots;
  private final boolean clusterIsClientMode;

  public DistributedPoliciesMapEntryListener(ApiService apiService,
                                             ApiTrackingService apiTrackingService,
                                             PolicySetDeploymentService policySetDeploymentService,
                                             ContractSnapshots contractSnapshots,
                                             boolean clusterIsClientMode) {
    this.apiService = apiService;
    this.apiTrackingService = apiTrackingService;
    this.policySetDeploymentService = policySetDeploymentService;
    this.contractSnapshots = contractSnapshots;
    this.clusterIsClientMode = clusterIsClientMode;
  }

  @Override
  public void entryAdded(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    ifDebug(() -> LOGGER.debug("Policies map entry added [{}] by {} [hashCode={}]", getApiKey(entry),
                               entry.getMember().getAddress(),
                               getConfigEntry(entry).hashCode()));

    if (clusterIsClientMode || !entry.getMember().localMember()) {
      ifDebug(() -> LOGGER.debug("Storing map entry [{}] into map store as a backup", getApiKey(entry)));

      handleEntry(entry);
    }
  }

  @Override
  public void entryUpdated(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    ifDebug(() -> LOGGER.debug("Policies map entry updated [{}] by {} [hashCode={}]", getApiKey(entry),
                               entry.getMember().getAddress(),
                               getConfigEntry(entry).hashCode()));

    if (clusterIsClientMode || !entry.getMember().localMember()) {
      ifDebug(() -> LOGGER.debug("Updating map entry [{}] into map store as a backup", getApiKey(entry)));

      handleEntry(entry);
    }
  }

  @Override
  public void entryRemoved(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    ifDebug(() -> LOGGER.debug("Policies map entry removed [{}].", getApiKey(entry)));

    if (clusterIsClientMode || !entry.getMember().localMember()) {
      ifDebug(() -> LOGGER.debug("Removing map entry [{}] as the requester is not us.", getApiKey(entry)));

      apiTrackingService.apiUntracked(getApiKey(entry));
    } else {
      ifDebug(() -> LOGGER.debug("Local member fired the removal of [{}].", getApiKey(entry)));
    }
  }

  private ApiKey getApiKey(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    return new ApiKey(Long.parseLong(entry.getKey()));
  }

  private DistributedApiConfigurationEntry getConfigEntry(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    return entry.getValue();
  }

  private void handleEntry(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    if (apiService.isDeployed(getApiKey(entry))) {
      policySetDeploymentService.policiesForApi(getApiKey(entry), getConfigEntry(entry).getPolicySet());
      contractSnapshots.slas(getApiKey(entry), getConfigEntry(entry).getSlas());
    } else {
      // If API is not present, then we just copy the state of the map to the file system
      policySetDeploymentService.conciliatePolicies(getApiKey(entry),
                                                    getConfigEntry(entry).getPolicySet().getPolicyDefinitions());
    }
  }

  private void ifDebug(Runnable closure) {
    if (LOGGER.isDebugEnabled()) {
      closure.run();
    }
  }
}
