/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment;

import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.CLIENT_SECRET;
import static com.mulesoft.mule.runtime.gw.api.time.period.Period.millis;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;
import static java.util.Arrays.asList;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;
import static org.mockito.Matchers.isA;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientConnectionListener;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.PlatformInteractionLifecycle;
import com.mulesoft.mule.runtime.gw.deployment.service.DefaultApiService;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTracker;
import com.mulesoft.mule.runtime.gw.deployment.tracking.RetriableApiTracker;
import com.mulesoft.mule.runtime.gw.metrics.GatewayMetricsAdapter;
import com.mulesoft.mule.runtime.gw.metrics.event.status.ApiRequestsTracker;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.FlexibleGateKeeper;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.HdpApisHealthCheckListener;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.StrictGateKeeper;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterCoreExtension;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterManager;

import java.util.Collection;
import java.util.List;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class ApiDeploymentCoreExtensionTestCase extends AbstractMuleTestCase {

  public static final String ENCRYPTION_SYSTEM_PROPERTY = "anypoint.platform.encryption_key";

  @Rule
  public SystemPropertyTemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  @Rule
  public SystemProperty clientId = new SystemProperty("anypoint.platform.client_id", "clientId");

  @Rule
  public SystemProperty clientSecret = new SystemProperty("anypoint.platform.client_secret", "clientSecret");

  private HazelcastClusterManager clusterManager;
  private DeploymentService deploymentService;
  private Registry registry;
  private ApiPlatformClient client;

  private ApiDeploymentCoreExtension coreExtension;

  public ApiDeploymentCoreExtensionTestCase(HazelcastClusterManager clusterManager) {
    this.clusterManager = clusterManager;
  }

  @Parameterized.Parameters
  public static Collection<Object[]> data() {
    return asList(new Object[][] {
        {mock(HazelcastClusterManager.class, RETURNS_DEEP_STUBS)},
        {null}
    });
  }

  @Before
  public void setUp() {
    HazelcastClusterCoreExtension clusterCoreExtension = mock(HazelcastClusterCoreExtension.class);
    ArtifactClassLoader artifactClassloader = mock(ArtifactClassLoader.class);
    deploymentService = mock(DeploymentService.class);
    registry = mock(Registry.class, RETURNS_DEEP_STUBS);
    client = mock(ApiPlatformClient.class);

    when(clusterCoreExtension.getHazelcastManager()).thenReturn(clusterManager);
    when(artifactClassloader.getClassLoader()).thenReturn(Thread.currentThread().getContextClassLoader());

    coreExtension = new ApiDeploymentCoreExtension();

    coreExtension.setClusterCoreExtension(clusterCoreExtension);
    coreExtension.setContainerClassLoader(artifactClassloader);

    injectDeploymentService(deploymentService);
    overrideExtensionLoaded();
    overrideContractService();
    overridePlatformClient();
  }

  @After
  public void tearDown() {
    System.clearProperty("anypoint.platform.gatekeeper");
    System.clearProperty("anypoint.platform.metrics_enabled");
  }

  @Test
  public void metricsDisabled() throws InitialisationException {
    setProperty("anypoint.platform.metrics_enabled", "false");

    coreExtension.initialiseCoreExtension();
    coreExtension.finishCoreExtensionInitialization();

    assertThat(getDeploymentListeners(), contains(instanceOf(HdpApisHealthCheckListener.class),
                                                  instanceOf(PolicySetDeploymentService.class),
                                                  instanceOf(OfflineModeApiDeploymentListener.class)));
    verify(deploymentService).addDeploymentListener(isA(DefaultApiService.class));
  }

  @Test
  public void gateKeeperDisabled() throws InitialisationException {
    setProperty("anypoint.platform.gatekeeper", "disabled");

    coreExtension.initialiseCoreExtension();
    coreExtension.finishCoreExtensionInitialization();

    assertThat(getDeploymentListeners(), contains(instanceOf(HdpApisHealthCheckListener.class),
                                                  instanceOf(PolicySetDeploymentService.class),
                                                  instanceOf(OfflineModeApiDeploymentListener.class),
                                                  instanceOf(GatewayMetricsAdapter.class),
                                                  instanceOf(ApiRequestsTracker.class)));
    verify(deploymentService).addDeploymentListener(isA(DefaultApiService.class));
  }

  @Test
  public void strictGateKeeperEnabled() throws InitialisationException {
    setProperty("anypoint.platform.gatekeeper", "strict");

    coreExtension.initialiseCoreExtension();
    coreExtension.finishCoreExtensionInitialization();

    assertThat(getDeploymentListeners(), contains(instanceOf(HdpApisHealthCheckListener.class),
                                                  instanceOf(PolicySetDeploymentService.class),
                                                  instanceOf(OfflineModeApiDeploymentListener.class),
                                                  instanceOf(GatewayMetricsAdapter.class),
                                                  instanceOf(ApiRequestsTracker.class)));

    coreExtension.startCoreExtension();
    coreExtension.initialiseRestClient();

    assertThat(getDeploymentListeners(), hasItem(instanceOf(StrictGateKeeper.class)));

    verify(deploymentService).addDeploymentListener(isA(DefaultApiService.class));
  }

  @Test
  public void flexibleGateKeeperEnabled() throws InitialisationException {
    setProperty("anypoint.platform.gatekeeper", "flexible");

    coreExtension.initialiseCoreExtension();
    coreExtension.finishCoreExtensionInitialization();

    assertThat(getDeploymentListeners(), contains(instanceOf(HdpApisHealthCheckListener.class),
                                                  instanceOf(PolicySetDeploymentService.class),
                                                  instanceOf(OfflineModeApiDeploymentListener.class),
                                                  instanceOf(GatewayMetricsAdapter.class),
                                                  instanceOf(ApiRequestsTracker.class)));

    coreExtension.startCoreExtension();
    coreExtension.initialiseRestClient();

    assertThat(getDeploymentListeners(), hasItem(instanceOf(FlexibleGateKeeper.class)));

    verify(deploymentService).addDeploymentListener(isA(DefaultApiService.class));
  }

  @Test
  public void flexibleGateKeeperEnabledByDefault() throws InitialisationException {

    coreExtension.initialiseCoreExtension();
    coreExtension.finishCoreExtensionInitialization();

    assertThat(getDeploymentListeners(), contains(instanceOf(HdpApisHealthCheckListener.class),
                                                  instanceOf(PolicySetDeploymentService.class),
                                                  instanceOf(OfflineModeApiDeploymentListener.class),
                                                  instanceOf(GatewayMetricsAdapter.class),
                                                  instanceOf(ApiRequestsTracker.class)));
    verify(deploymentService).addDeploymentListener(isA(DefaultApiService.class));

    coreExtension.startCoreExtension();

    assertThat(getDeploymentListeners(), hasItem(instanceOf(FlexibleGateKeeper.class)));
  }

  @Test
  public void onFirstAppDeployed() throws InitialisationException {
    coreExtension.initialiseCoreExtension();
    coreExtension.finishLazyInitialization();

    assertThat(getDeploymentListeners(), contains(instanceOf(HdpApisHealthCheckListener.class),
                                                  instanceOf(PolicySetDeploymentService.class),
                                                  instanceOf(GatewayMetricsAdapter.class),
                                                  instanceOf(ApiRequestsTracker.class),
                                                  instanceOf(FlexibleGateKeeper.class),
                                                  instanceOf(RetriableApiTracker.class)));
    verify(deploymentService).addDeploymentListener(isA(DefaultApiService.class));
    List<ApiPlatformClientConnectionListener> connectionListeners = getConnectionListeners();
    assertThat(connectionListeners, hasSize(5));
    assertThat(connectionListeners.get(0), instanceOf(PlatformInteractionLifecycle.class));
    assertThat(connectionListeners.get(1), instanceOf(PlatformInteractionLifecycle.class));
    assertThat(connectionListeners.get(2), instanceOf(PlatformInteractionLifecycle.class));

  }

  @Test
  public void onFirstAppDeployedAndClientInitFailure() throws InitialisationException {
    when(client.connect()).thenThrow(new RuntimeException());
    coreExtension.initialiseCoreExtension();
    coreExtension.finishLazyInitialization();

    assertThat(getDeploymentListeners(), contains(instanceOf(HdpApisHealthCheckListener.class),
                                                  instanceOf(PolicySetDeploymentService.class),
                                                  instanceOf(OfflineModeApiDeploymentListener.class),
                                                  instanceOf(GatewayMetricsAdapter.class),
                                                  instanceOf(ApiRequestsTracker.class),
                                                  instanceOf(FlexibleGateKeeper.class)));
    verify(deploymentService).addDeploymentListener(isA(DefaultApiService.class));

    List<ApiPlatformClientConnectionListener> connectionListeners = getConnectionListeners();
    assertThat(connectionListeners, hasSize(5));
    assertThat(connectionListeners.get(0), instanceOf(PlatformInteractionLifecycle.class));
    assertThat(connectionListeners.get(1), instanceOf(PlatformInteractionLifecycle.class));
    assertThat(connectionListeners.get(2), instanceOf(PlatformInteractionLifecycle.class));
  }

  @Test
  public void policySetDeploymentServiceRetrierWithNoDelay() throws InitialisationException {
    coreExtension.initialiseCoreExtension();
    coreExtension.finishCoreExtensionInitialization();

    SchedulingConfiguration retrierSchedulingConfiguration = policySetDeploymentServiceRetrierConfiguration();

    assertThat(retrierSchedulingConfiguration.delay(), is(millis(0)));
    assertThat(retrierSchedulingConfiguration.frequency(), is(millis(0)));
  }

  @Test
  public void policySetDeploymentServiceRetrierThreadName() throws InitialisationException {
    coreExtension.initialiseCoreExtension();
    coreExtension.finishCoreExtensionInitialization();

    assertThat(retrierThreadName(), is("agw-policy-set-deployment"));
  }

  @Test
  public void removeSecretsWhenStoppingRuntime() throws MuleException {
    setProperty(ENCRYPTION_SYSTEM_PROPERTY, "GatewayTeamKey00");
    setProperty(CLIENT_SECRET, "SuperSecretClientSecret");

    coreExtension.stop();

    assertThat(getProperty(ENCRYPTION_SYSTEM_PROPERTY, null), is(nullValue()));
    assertThat(getProperty(CLIENT_SECRET, null), is(nullValue()));
  }

  private String retrierThreadName() {
    return new Inspector(coreExtension).read("policySetDeploymentService.runnableRetrier.threadName");
  }

  private SchedulingConfiguration policySetDeploymentServiceRetrierConfiguration() {
    return new Inspector(coreExtension).read("policySetDeploymentService.runnableRetrier.initialSchedulingConfiguration");
  }

  private List<ApiDeploymentListener> getDeploymentListeners() {
    return new Inspector(coreExtension).read("apiService.apiNotificationManager.apiDeploymentListeners");
  }

  private List<ApiPlatformClientConnectionListener> getConnectionListeners() {
    return new Inspector(coreExtension).read("clientProvider.connectionListeners");
  }

  private void injectDeploymentService(DeploymentService deploymentService) {
    overrideVariable("deploymentService").in(coreExtension).with(deploymentService);
  }

  private void overridePlatformClient() {
    overrideVariable("clientProvider.client").in(coreExtension).with(client);
  }

  private void overrideExtensionLoaded() {
    overrideVariable("extensionLoaded").in(coreExtension).with(true);
  }

  private void overrideContractService() {
    overrideVariable("contractService").in(coreExtension).with(mock(ContractService.class));
  }

}
