/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertThat;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.autodiscovery.ByIdAutodiscovery;
import com.mulesoft.mule.runtime.gw.autodiscovery.api.Autodiscovery;
import com.mulesoft.mule.runtime.gw.deployment.service.DefaultApiService;
import com.mulesoft.mule.runtime.gw.hdp.config.HighDensityProxyConfiguration;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.hdp.ApiRecordDto;
import com.mulesoft.mule.runtime.gw.model.hdp.ApiRegistry;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class ApiServiceTestCase extends AbstractMuleTestCase {

  @Captor
  private ArgumentCaptor<Api> apiCaptor;

  @Captor
  private ArgumentCaptor<ApiImplementation> implementationCaptor;

  private ApiDeploymentListener deploymentListener;
  private DeploymentService deploymentService;
  private MockLogger logger;

  private DefaultApiService apiService;

  private HighDensityProxyConfiguration hdpConfiguration = new HighDensityProxyConfiguration();

  public static final String HDP_APP = "service-mesh";

  @Before
  public void setUp() {
    deploymentListener = mock(ApiDeploymentListener.class);
    deploymentService = mock(DeploymentService.class);
    logger = new MockLogger();

    apiService = new DefaultApiService(deploymentService);

    overrideLogger().in(apiService).with(logger);
  }

  @Test
  public void onDeploymentStart() {
    Registry registry = mock(Registry.class, RETURNS_DEEP_STUBS);
    injectApiToContext(registry, API_KEY, API_KEY_2);
    apiService.addDeploymentListener(deploymentListener);

    apiService.onArtifactInitialised(APP, registry);

    assertThat(apiService.isDeployed(API_KEY), is(true));
    assertThat(apiService.isDeployed(API_KEY_2), is(true));
    verify(deploymentListener, times(2)).onApiDeploymentStart(apiCaptor.capture());
    assertThat(apiCaptor.getAllValues(), hasSize(2));
    assertThat(apiCaptor.getAllValues().get(0).getKey(), is(API_KEY));
    assertThat(apiCaptor.getAllValues().get(1).getKey(), is(API_KEY_2));
  }

  @Test
  public void onDeploymentStartSameApiTwiceSecondIsIgnored() {
    Application application1 = mock(Application.class);
    Application application2 = mock(Application.class);
    Registry registry1 = mock(Registry.class, RETURNS_DEEP_STUBS);
    Registry registry2 = mock(Registry.class, RETURNS_DEEP_STUBS);
    injectApiToContext(registry1, API_KEY);
    injectApiToContext(registry2, API_KEY);
    when(deploymentService.findApplication(APP)).thenReturn(application1);
    when(deploymentService.findApplication(APP_2)).thenReturn(application2);

    apiService.onArtifactInitialised(APP, registry1);
    apiService.onArtifactInitialised(APP_2, registry2);

    assertThat(apiService.isDeployed(API_KEY), is(true));
    assertThat(apiService.getImplementation(API_KEY).get().getArtifactName(), is(application1.getArtifactName()));
  }

  @Test
  public void onDeploymentSuccess() {
    initialiseApiService();
    apiService.onDeploymentSuccess(APP);

    assertThat(apiService.isDeployed(API_KEY), is(true));
    verify(deploymentListener, times(1)).onApiDeploymentSuccess(apiCaptor.capture());
    assertThat(apiCaptor.getAllValues(), hasSize(1));
    assertThat(apiCaptor.getAllValues().get(0).getKey(), is(API_KEY));
  }

  @Test
  public void onUndeploymentStart() {
    initialiseApiService();
    Api spiedApi = spyApi();

    apiService.onUndeploymentStart(APP);

    assertThat(apiService.isDeployed(API_KEY), is(false));
    verify(deploymentListener, times(1)).onApiUndeploymentStart(implementationCaptor.capture());
    assertThat(implementationCaptor.getAllValues(), hasSize(1));
    assertThat(implementationCaptor.getAllValues().get(0).getApiKey(), is(API_KEY));
    verify(spiedApi).dispose();
    assertThat(logger.lines().get(logger.lines().size() - 1),
               is(new DebugLine("API {} un-deployment started", spiedApi)));
  }

  @Test
  public void apiIsRemovedFromServiceAfterNotifyingListeners() {
    Application application = injectApplication(APP);
    ApiDeploymentListener mockListener = new ApiDeploymentListener() {

      @Override
      public void onApiUndeploymentStart(ApiImplementation implementation) {
        assertThat(apiService.get(API_KEY), not(empty()));
      }
    };
    apiService.addDeploymentListener(mockListener);
    apiService.onArtifactInitialised(APP, application.getRegistry());

    apiService.onUndeploymentStart(APP);
  }

  @Test
  public void onRedeploymentStart() {
    initialiseApiService();
    Api spiedApi = spyApi();

    apiService.onRedeploymentStart(APP);

    assertRedeploy(spiedApi);
  }

  @Test
  public void onRedeployUndeployNotificationDoesNotFire() {
    initialiseApiService();
    Api spiedApi = spyApi();

    apiService.onRedeploymentStart(APP);
    apiService.onUndeploymentStart(APP);

    assertRedeploy(spiedApi);
    verify(deploymentListener, times(0)).onApiUndeploymentStart(any());
  }

  @Test
  public void onHdpApiUpdateCreate() {
    createHdpApis();
    verify(deploymentListener, times(2)).onApiDeploymentSuccess(apiCaptor.capture());
    assertThat(apiCaptor.getAllValues(), hasSize(2));
    assertThat(apiCaptor.getAllValues().get(0).getKey().id(), is(1001L));
    assertThat(apiCaptor.getAllValues().get(0).getImplementation().getHdpService().orElse(null), is("one"));
    assertThat(apiCaptor.getAllValues().get(1).getKey().id(), is(1002L));
    assertThat(apiCaptor.getAllValues().get(1).getImplementation().getHdpService().orElse(null), is("two"));
  }

  @Test
  public void onHdpApiUpdateRemove() {
    ApiRegistry apiRegistry = createHdpApis();
    apiRegistry.getApiRecords().remove(0);
    apiService.updateHdpApis(apiRegistry);

    verify(deploymentListener, times(1)).onApiUndeploymentStart(implementationCaptor.capture());
    assertThat(implementationCaptor.getAllValues(), hasSize(1));
    assertThat(implementationCaptor.getAllValues().get(0).getApiKey().id(), is(1001L));
  }

  @Test
  public void onHdpApiUpdateRenameService() {
    ApiRegistry apiRegistry = createHdpApis();
    apiRegistry.getApiRecords().remove(1);
    apiRegistry.getApiRecords().add(new ApiRecordDto(1002, "two-renamed"));
    apiService.updateHdpApis(apiRegistry);

    verify(deploymentListener, times(1)).onApiUndeploymentStart(implementationCaptor.capture());
    assertThat(implementationCaptor.getAllValues(), hasSize(1));
    assertThat(implementationCaptor.getAllValues().get(0).getApiKey().id(), is(1002L));

    verify(deploymentListener, times(3)).onApiDeploymentSuccess(apiCaptor.capture());
    assertThat(apiCaptor.getAllValues(), hasSize(3));
    assertThat(apiCaptor.getAllValues().get(2).getKey().id(), is(1002L));
    assertThat(apiCaptor.getAllValues().get(2).getImplementation().getHdpService().orElse(null), is("two-renamed"));

  }

  private ApiRegistry createHdpApis() {
    apiService.addDeploymentListener(deploymentListener);
    Application application = injectHdpApplication();
    when(application.getRegistry().lookupByName("hdp-apis-healthcheck")).thenReturn(Optional.empty());
    apiService.onDeploymentSuccess(HDP_APP);

    List<ApiRecordDto> apis = new ArrayList<>();
    apis.add(new ApiRecordDto(1001, "one"));
    apis.add(new ApiRecordDto(1002, "two"));

    ApiRegistry apiRegistry = new ApiRegistry(HDP_APP, apis);
    apiService.updateHdpApis(apiRegistry);
    verify(deploymentListener, times(2)).onApiDeploymentStart(any());
    return apiRegistry;
  }

  private void initialiseApiService() {
    Application application = injectApplication(APP);
    apiService.addDeploymentListener(deploymentListener);
    apiService.onArtifactInitialised(APP, application.getRegistry());
  }

  private void assertRedeploy(Api spiedApi) {
    assertThat(apiService.isDeployed(API_KEY), is(false));
    verify(deploymentListener, times(1)).onApiRedeploymentStart(implementationCaptor.capture());
    assertThat(implementationCaptor.getAllValues(), hasSize(1));
    assertThat(implementationCaptor.getAllValues().get(0).getApiKey(), is(API_KEY));
    verify(spiedApi, times(0)).dispose();
    assertThat(logger.lines().get(logger.lines().size() - 1),
               is(new DebugLine("API {} re-deployment started", spiedApi)));
  }

  private Api spyApi() {
    Map<ApiKey, Api> serviceApis = peekApis(apiService);
    Api spiedApi = spy(serviceApis.get(API_KEY));
    serviceApis.put(API_KEY, spiedApi);
    return spiedApi;
  }

  private Map<ApiKey, Api> peekApis(DefaultApiService apiService) {
    return new Inspector(apiService).read("apis");
  }

  private Application injectHdpApplication() {
    Application application = injectApplication(HDP_APP);
    ConfigurationProperties config = mock(ConfigurationProperties.class);
    when(config.resolveBooleanProperty(hdpConfiguration.getHdpApplicationProperty())).thenReturn(of(true));
    when(application.getRegistry().lookupByType(ConfigurationProperties.class)).thenReturn(of(config));
    return application;
  }

  private Application injectApplication(String appName) {
    Application application = mock(Application.class, RETURNS_DEEP_STUBS);
    injectApiToContext(application.getRegistry(), API_KEY);
    when(deploymentService.findApplication(appName)).thenReturn(application);
    when(application.getArtifactName()).thenReturn(appName);
    when(application.getRegistry().lookupByType(ConfigurationProperties.class)).thenReturn(empty());
    return application;
  }

  private void injectApiToContext(Registry registry, ApiKey... apiKeys) {
    List<ByIdAutodiscovery> metadatas = newArrayList(apiKeys).stream()
        .map(apiKey -> {
          ByIdAutodiscovery metadata = new ByIdAutodiscovery();
          metadata.setId(apiKey.id());
          return metadata;
        }).collect(Collectors.toList());

    when(registry.lookupAllByType(Autodiscovery.class))
        .thenReturn(newArrayList(metadatas));
    when(registry.lookupByType(Flow.class)).thenReturn(Optional.of(mock(Flow.class)));
  }
}
