/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.contracts;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.module.deployment.api.DeploymentService;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.service.DefaultApiService;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.deployment.tracking.DefaultApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiTrackingInfo;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import java.util.Map;

import org.junit.Before;
import org.junit.Test;

public class ApiContractSupplierTestCase {

  private ApiService apiService;
  private Api api;
  private Api anotherApi;
  private ContractService contractService;
  private ApiTrackingService apiTrackingService;

  @Before
  public void setUp() {
    apiService = new DefaultApiService(mock(DeploymentService.class));
    contractService = mock(ContractService.class);
    apiTrackingService = new DefaultApiTrackingService(apiService, null, null, null, contractService, null);
    api = new Api(key(), null, null);
    anotherApi = new Api(anotherKey(), null, null);
  }

  @Test
  public void emptyTrackedAndContracts() {
    assertThat(apiTrackingService.getTrackedApis(), empty());
    assertThat(apiTrackingService.getTrackedApisRequiringContracts(), empty());
  }

  @Test
  public void checkApisNotTracked() {
    add(api, anotherApi);

    assertThat(apiTrackingService.getTrackedApis(), empty());
    assertThat(apiTrackingService.getTrackedApisRequiringContracts(), empty());
  }

  @Test
  public void checkTrackedApiButWithoutContracts() {
    addTwoApisAndSetOneAsTracked();

    assertThat(apiTrackingService.getTrackedApis(), is(asList(api)));
    assertThat(apiTrackingService.getTrackedApisRequiringContracts(), empty());
  }

  @Test
  public void checkWithContractsButNotTracked() {
    add(api);
    track(api.getKey());

    assertThat(apiTrackingService.getTrackedApis(), empty());
    assertThat(apiTrackingService.getTrackedApisRequiringContracts(), empty());
  }

  @Test
  public void trackedWithContracts() {
    addTwoApisAndSetOneAsTracked();
    track(api.getKey());

    assertThat(apiTrackingService.getTrackedApis(), is(asList(api)));
    assertThat(apiTrackingService.getTrackedApisRequiringContracts(), is(asList(api)));
  }

  @Test
  public void trackMultipleApis() {
    add(api, anotherApi);
    track(api.getKey(), anotherApi.getKey());
    api.updateTrackingInfo(tracked());
    anotherApi.updateTrackingInfo(tracked());

    assertThat(apiTrackingService.getTrackedApis(), is(asList(anotherApi, api)));
    assertThat(apiTrackingService.getTrackedApisRequiringContracts(), is(asList(anotherApi, api)));
  }

  private void addTwoApisAndSetOneAsTracked() {
    add(api, anotherApi);
    api.updateTrackingInfo(tracked());
  }

  private ApiTrackingInfo tracked() {
    return new ApiTrackingInfo(1L, "", "", "",
                               "", "", "", "", 99,
                               "", "", "", "");
  }

  private ApiContractSupplierTestCase track(ApiKey... keys) {
    when(contractService.trackedApis()).thenReturn(asList(keys));
    return this;
  }

  private ApiContractSupplierTestCase add(Api... apis) {
    for (Api api : apis) {
      Map<ApiKey, Api> apisWithinService = new Inspector(apiService).read("apis");
      apisWithinService.put(api.getKey(), api);
      overrideVariable("apis").in(apiService).with(apisWithinService);
    }
    return this;
  }

  private ApiKey anotherKey() {
    return API_KEY_2;
  }

  private ApiKey key() {
    return API_KEY;
  }
}
