/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.contracts;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.mule.runtime.gw.api.contract.Contract.builder;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;

import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.contract.NoSla;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.deployment.ApiPlatformTestCase;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.contracts.DefaultClientFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.Lock;

import org.junit.Before;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;

public abstract class ContractSnapshotsTestCase extends ApiPlatformTestCase {

  static final List<Contract> EMPTY_UPDATE = new ArrayList<>();
  private static final Long INEXISTENT_API_ID = 100L;

  ContractSnapshots snapshots;
  Lock lock;

  @Captor
  private ArgumentCaptor<List<Contract>> captor;

  @Before
  @Override
  public void setUp() throws Exception {
    super.setUp();
    mocks.singleTrackedApi();
    lock = mock(Lock.class);
    snapshots = new ContractSnapshots(mocks.apiService(), lock, new DefaultClientFactory());
    ApiContracts apiContracts = mocks.api().getContracts();
    when(apiContracts.contractsLoaded()).thenReturn(true);
    when(mocks.apiService().get(API_KEY)).thenReturn(of(mocks.api()));
    when(mocks.apiService().getContracts(API_KEY)).thenReturn(ofNullable(apiContracts));
    when(mocks.apiService().getContracts(inexistentApi())).thenReturn(empty());
    when(mocks.apiService().get(inexistentApi())).thenReturn(empty());
  }

  void checkContractsUpdates(List<Contract>... expectedUpdates) {
    checkContractsUpdates(mocks.api(), expectedUpdates);
  }

  void checkContractsUpdates(Api api, List<Contract>... expectedUpdates) {
    int contractsCount = expectedUpdates.length;
    verify(api.getContracts(), times(contractsCount)).updateContracts(captor.capture());

    assertThat(captor.getAllValues(), hasSize(contractsCount));

    for (int i = 0; i < contractsCount; i++) {
      assertThat("Failed on list ith=" + i, captor.getAllValues().get(i), is(expectedUpdates[i]));
    }
  }

  void verifyApiTestInvocations() {
    verify(mocks.api(), times(3)).getTrackingInfo();
    verify(mocks.api(), times(3)).getImplementation();
    verify(mocks.api(), atLeastOnce()).getContracts();
    verifyNoMoreInteractions(mocks.api());
  }

  Contract contractMultiLimit(int anotherTierId) {
    return contract(slaWithMultipleLimits(anotherTierId),
                    client(anotherTierId));
  }

  Contract contractSingleLimit(int slaId) {
    return contract(sla(slaId), client(slaId));
  }

  Contract noSlaContractForAnotherClient(String slaId) {
    return builder().withClient(anotherClient(slaId)).withSla(new NoSla()).build();
  }

  Contract noSlaContractForAnotherClient(int slaId) {
    return noSlaContractForAnotherClient(Integer.toString(slaId));
  }

  Contract noSlaContractForClient(String slaId) {
    return builder().withClient(client(slaId)).build();
  }

  Contract noSlaContractForClient(int slaId) {
    return noSlaContractForClient(Integer.toString(slaId));
  }

  Contract noSlaContractForThirdClient(int slaId) {
    return builder().withClient(aThirdClient(slaId)).withSla(new NoSla()).build();
  }

  protected Contract contract(Sla sla, Client client) {
    return Contract.builder().withClient(client).withSla(sla).build();
  }

  protected Client client(int slaId) {
    return client(Integer.toString(slaId));
  }

  protected Client client(String slaId) {
    return client(0, slaId);
  }

  protected Client client(int id, int slaId) {
    return client(id, Integer.toString(slaId));
  }

  protected Client client(int id, String slaId) {
    return validClient(platformContractAdapterFromDto(dtoClient(id, 1L, slaId)));
  }

  protected Client anotherClient(int slaId) {
    return anotherClient(Integer.toString(slaId));
  }

  protected Client anotherClient(String slaId) {
    return validClient(platformContractAdapterFromDto(dtoClient(1, 2L, slaId)));
  }

  protected Client aThirdClient(int slaId) {
    return validClient(platformContractAdapterFromDto(dtoClient(2, 2L, Integer.toString(slaId))));
  }

  protected ApiKey inexistentApi() {
    return new ApiKey(INEXISTENT_API_ID);
  }

  protected PlatformContractAdapter platformContractAdapterFromDto(ApiClientDto client) {
    return new PlatformContractAdapter(client);
  }

  protected PlatformContractAdapter platformContractAdapter(int slaId) {
    return platformContractAdapter(Integer.toString(slaId));
  }

  protected PlatformContractAdapter platformContractAdapter(String slaId) {
    return platformContractAdapter(0, slaId);
  }

  protected PlatformContractAdapter platformContractAdapter(int id, String slaId) {
    return platformContractAdapterFromDto(dtoClient(id, 1L, slaId));
  }

  protected PlatformContractAdapter anotherPlatformContractAdapter(int slaId) {
    return anotherPlatformContractAdapter(Integer.toString(slaId));
  }

  protected PlatformContractAdapter anotherPlatformContractAdapter(String slaId) {
    return platformContractAdapterFromDto(dtoClient(1, 2L, slaId));
  }

  protected PlatformContractAdapter aThirdPlatformContractAdapter(int slaId) {
    return platformContractAdapterFromDto(dtoClient(2, 2L, Integer.toString(slaId)));
  }

  private Client validClient(PlatformContractAdapter platformContractAdapter) {
    return new DefaultClientFactory().create(platformContractAdapter.clientId(), platformContractAdapter.clientSecret(),
                                             platformContractAdapter.clientName());
  }

}
