/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.mocks;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.construct.Flow;

import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.session.BackoffBarrier;
import com.mulesoft.anypoint.backoff.session.SessionMetadata;
import com.mulesoft.mule.runtime.gw.client.model.ApiClientsResponse;
import com.mulesoft.mule.runtime.gw.client.model.ApiResponse;
import com.mulesoft.mule.runtime.gw.client.session.ApiPlatformSession;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients.PlatformClientsRetriever;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ApiKeepAliveRunnable;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ApisRunnable;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ClientsRunnable;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;

import java.util.ArrayList;
import java.util.List;

public class ApiPlatformMocks {

  public static final String ORG_ID = "orgId";
  public static final String ENV_ID = "envId";

  private final ApiPlatformSession platformSession;
  private final ApiService apiService;
  private final ApiPlatformSessionFactory platformSessionFactory;
  private final BackoffBarrier backoffBarrier;
  private final BackoffConfiguration backoffConfiguration;
  private Api api;
  private Api secondApi;
  private ApiTrackingService apiTrackingService;

  public ApiPlatformMocks() {
    apiTrackingService = mock(ApiTrackingService.class);
    apiService = mock(ApiService.class);
    api = mock(Api.class);
    when(api.getContracts()).thenReturn(mock(ApiContracts.class));

    secondApi = mock(Api.class);
    platformSessionFactory = mock(ApiPlatformSessionFactory.class);
    platformSession = mock(ApiPlatformSession.class);
    when(platformSessionFactory.create()).thenReturn(platformSession);
    when(platformSession.metadata()).thenReturn(emptySessionMetadata());
    backoffBarrier = (sessionMetadata, configuration) -> false;
    backoffConfiguration = new BackoffConfiguration.Builder(true).build();

    createExpectations();
  }

  private SessionMetadata emptySessionMetadata() {
    return new SessionMetadata() {

      @Override
      public int requests() {
        return 0;
      }

      @Override
      public List<Integer> statusCodes() {
        return new ArrayList<>();
      }

      @Override
      public int getCount(int statusCode) {
        return 0;
      }
    };
  }

  public ClientsRunnable clientsRunnable() {
    return new ClientsRunnable(apiTrackingService, platformSessionFactory, backoffBarrier, backoffConfiguration,
                               new PlatformClientsRetriever(
                                                            platformSessionFactory, apiTrackingService));
  }

  public ApisRunnable apisRunnable() {
    return new ApisRunnable(apiTrackingService, platformSessionFactory, backoffBarrier, backoffConfiguration);
  }

  public ApiKeepAliveRunnable keepAliveRunnable() {
    return new ApiKeepAliveRunnable(apiTrackingService,
                                    platformSessionFactory,
                                    backoffBarrier, backoffConfiguration);
  }

  public ApiPlatformSession session() {
    return platformSession;
  }

  public ApiService apiService() {
    return apiService;
  }

  public ApiTrackingService apiTrackingService() {
    return apiTrackingService;
  }

  public ApiPlatformMocks singleTrackedApi() {
    when(apiTrackingService.getTrackedApis()).thenReturn(newArrayList(api()));
    return this;
  }

  public ApiPlatformMocks multipleTrackedApis() {
    when(apiTrackingService.getTrackedApis()).thenReturn(newArrayList(api(), secondApi()));
    return this;
  }

  public ApiPlatformMocks singleApiFetchContracts() {
    singleTrackedApi();
    when(apiTrackingService.getTrackedApisRequiringContracts()).thenReturn(newArrayList(api()));
    return this;
  }

  public ApiPlatformMocks multipleApisFetchContracts() {
    multipleTrackedApis();
    when(apiTrackingService.getTrackedApisRequiringContracts()).thenReturn(newArrayList(api(), secondApi()));
    return this;
  }

  public ApiPlatformMocks stoppedApi() {
    when(api.getImplementation().getFlow().getMuleContext().isStarted()).thenReturn(false);
    return this;
  }

  public Api api() {
    return api;
  }

  public Api secondApi() {
    return secondApi;
  }

  public ApiPlatformMocks getApiClientsReturns(Api api, ApiClientsResponse apiClientsResponse) {
    TrackingInfo trackingInfo = api.getTrackingInfo();
    when(platformSession.getApiClients(trackingInfo.getOrganizationId(), trackingInfo.getEnvironmentId(), trackingInfo.getId(),
                                       trackingInfo.getContractsEntityTag()))
                                           .thenReturn(apiClientsResponse);
    return this;
  }

  public ApiPlatformMocks getApiClientsThrows(Api api, Exception exception) {
    TrackingInfo trackingInfo = api.getTrackingInfo();
    when(platformSession.getApiClients(trackingInfo.getOrganizationId(), trackingInfo.getEnvironmentId(), trackingInfo.getId(),
                                       trackingInfo.getContractsEntityTag()))
                                           .thenThrow(exception);
    return this;
  }

  public ApiPlatformMocks getApiReturns(Api api, ApiResponse apiResponse) {
    when(platformSession.getApi(api)).thenReturn(apiResponse);
    return this;
  }

  public ApiPlatformMocks getApiThrows(Api api, Exception exception) {
    when(platformSession.getApi(api)).thenThrow(exception);
    return this;
  }

  private void createExpectations() {
    when(api.getKey()).thenReturn(API_KEY);
    when(api.getTrackingInfo()).thenReturn(mock(TrackingInfo.class));
    when(api.getTrackingInfo().getId()).thenReturn(API_KEY.id());
    when(api.getTrackingInfo().getEnvironmentId()).thenReturn(ENV_ID);
    when(api.getTrackingInfo().getOrganizationId()).thenReturn(ORG_ID);
    when(api.getContracts()).thenReturn(mock(ApiContracts.class));
    when(api.getImplementation()).thenReturn(mock(ApiImplementation.class));
    when(api.getImplementation().getFlow()).thenReturn(mock(Flow.class));
    when(api.getImplementation().getFlow().getMuleContext()).thenReturn(mock(MuleContext.class));
    when(api.getImplementation().getFlow().getMuleContext().isStarted()).thenReturn(true);

    when(secondApi.getKey()).thenReturn(API_KEY_2);
    when(secondApi.getTrackingInfo()).thenReturn(mock(TrackingInfo.class));
    when(secondApi.getTrackingInfo().getId()).thenReturn(API_KEY_2.id());
    when(secondApi.getTrackingInfo().getEnvironmentId()).thenReturn(ENV_ID);
    when(secondApi.getTrackingInfo().getOrganizationId()).thenReturn(ORG_ID);
    when(secondApi.getContracts()).thenReturn(mock(ApiContracts.class));
    when(secondApi.getImplementation()).thenReturn(mock(ApiImplementation.class));
    when(secondApi.getImplementation().getFlow()).thenReturn(mock(Flow.class));
    when(secondApi.getImplementation().getFlow().getMuleContext()).thenReturn(mock(MuleContext.class));
    when(secondApi.getImplementation().getFlow().getMuleContext().isStarted()).thenReturn(true);
  }
}
