/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.pollers;

import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.DEFAULT_OUTAGE_STATUS_CODES;
import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.OUTAGE_STATUS_CODES;
import static com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration.DEFAULT_BACKOFF_STEPS;
import static com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration.DEFAULT_BACKON_STEPS;
import static com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration.DEFAULT_FAILURE_PERCENTAGE;
import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfigurationSupplier;
import com.mulesoft.anypoint.backoff.scheduler.BackoffScheduler;
import com.mulesoft.anypoint.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.anypoint.backoff.scheduler.runnable.BackoffRunnable;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.GatewayPoller;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.apis.GatewayApisPoller;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients.GatewayClientsPoller;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients.PlatformClientsRetriever;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.keepalive.GatewayKeepAlivePoller;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

@RunWith(Parameterized.class)
public class GatewayPollerTestCase extends GatewayPollersTestCase {

  private Function<BackoffSchedulerFactory, GatewayPoller> pollerManagerSupplier;
  private GatewayPoller pollerManager;
  private BackoffScheduler scheduler;

  @Parameters
  public static Collection<Object[]> parameters() {
    return asList(new Object[][] {
        {(Function<BackoffSchedulerFactory, GatewayPoller>) (factory) -> new GatewayApisPoller(
                                                                                               new GatewayConfiguration(),
                                                                                               mock(ApiTrackingService.class),
                                                                                               mock(ApiPlatformSessionFactory.class),
                                                                                               factory,
                                                                                               new BackoffConfigurationSupplier())},
        {(Function<BackoffSchedulerFactory, GatewayPoller>) (factory) -> new GatewayKeepAlivePoller(
                                                                                                    new GatewayConfiguration(),
                                                                                                    mock(ApiTrackingService.class),
                                                                                                    mock(ApiPlatformSessionFactory.class),
                                                                                                    factory,
                                                                                                    new BackoffConfigurationSupplier())},
        {(Function<BackoffSchedulerFactory, GatewayPoller>) (factory) -> new GatewayClientsPoller(
                                                                                                  new GatewayConfiguration(),
                                                                                                  mock(ApiTrackingService.class),
                                                                                                  mock(ApiPlatformSessionFactory.class),
                                                                                                  factory,
                                                                                                  new BackoffConfigurationSupplier(),
                                                                                                  mock(PlatformClientsRetriever.class))}
    });
  }


  public GatewayPollerTestCase(Function<BackoffSchedulerFactory, GatewayPoller> pollerManagerSupplier) {
    this.pollerManagerSupplier = pollerManagerSupplier;
  }

  @Override
  public void setUp() {
    super.setUp();
    BackoffSchedulerFactory factory = mock(BackoffSchedulerFactory.class);
    scheduler = mock(BackoffScheduler.class);
    when(factory.create(any())).thenReturn(scheduler);
    this.pollerManager = pollerManagerSupplier.apply(factory);
  }

  @Test
  public void runnablesWithDefaultStatusCodes() {
    assertThat(configuration(pollerManager.schedule()).statusCodes(), is(DEFAULT_OUTAGE_STATUS_CODES));
  }

  @Test
  public void runnablesWithCustomStatusCodes() {
    List<Integer> configuredStatusCodes = asList(522, 429, 666);

    System.setProperty(OUTAGE_STATUS_CODES, "522, 429, 666");

    assertThat(configuration(pollerManager.schedule()).statusCodes(), is(configuredStatusCodes));
  }

  @Test
  public void runnablesBackoffConfiguration() {
    assertDefault(configuration(pollerManager.schedule()));
  }

  @Test
  public void shutdownWhenPollerNotStarted() {
    pollerManager.shutdown();
    verifyNoMoreInteractions(scheduler);
  }

  @Test
  public void shutdownWhenDisposed() {
    pollerManager.schedule();
    pollerManager.shutdown();
    verify(scheduler).dispose();
  }

  private void assertDefault(BackoffConfiguration configuration) {
    assertThat(configuration.backoffSteps(), is(DEFAULT_BACKOFF_STEPS));
    assertThat(configuration.backonSteps(), is(DEFAULT_BACKON_STEPS));
    assertThat(configuration.failurePercentage(), is(DEFAULT_FAILURE_PERCENTAGE));
    assertThat(configuration.backoff(), is(defaultBackoffFunction()));
    assertThat(configuration.backon(), is(defaultBackonFunction()));
    assertThat(configuration.statusCodes(), is(DEFAULT_OUTAGE_STATUS_CODES));
  }

  private BackoffConfiguration configuration(BackoffRunnable pollerRunnable) {
    return new Inspector(pollerRunnable).read("currentState.configuration");
  }
}
