/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.service;

import static java.lang.String.format;
import static java.util.stream.IntStream.range;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.anyString;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.client.ClientsRepositoryException;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.dto.CoreServicesClientDto;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.response.PlatformResponse;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class CoreServicesClientsRepositoryTestCase {

  @Rule
  public ExpectedException thrown = ExpectedException.none();

  private static final String CLIENT_ID = "id";
  private ApiPlatformClient platform;
  private CoreServicesClientsRepository repository;
  private CoreServicesClientDto dto;

  @Before
  public void setUp() {
    this.platform = mock(ApiPlatformClient.class);
    this.repository = new CoreServicesClientsRepository(platform);
    this.dto = new CoreServicesClientDto(CLIENT_ID, "secret", "name", null, null);
  }

  @Test
  public void successfulGetClient() throws ClientsRepositoryException {
    range(200, 299).forEach(statusCode -> {
      when(platform.getClient(eq(CLIENT_ID))).thenReturn(new PlatformResponse<>(dto, statusCode));

      try {
        assertThat(repository.getBy(CLIENT_ID), is(adapt(dto)));
      } catch (ClientsRepositoryException e) {
        fail("Exception was raised.");
      }
    });
  }

  @Test
  public void errorStatusCode() {
    range(400, 600).forEach(statusCode -> {
      when(platform.getClient(eq(CLIENT_ID))).thenReturn(new PlatformResponse<>(dto, statusCode));

      boolean exceptionRaised = false;
      try {
        repository.getBy(CLIENT_ID);
      } catch (ClientsRepositoryException e) {
        exceptionRaised = true;
        assertThat(e.getStatus(), is(statusCode));
        assertThat(e.toString(),
                   is(format("Invalid client with id '%s'. (%s).", CLIENT_ID, statusCode)));
      }

      if (!exceptionRaised) {
        fail("Exception was not raised for status code " + statusCode);
      }
    });
  }

  @Test
  public void platformException() throws ClientsRepositoryException {
    int statusCode = 404;

    HttpResponseException responseException = mock(HttpResponseException.class);
    when(responseException.statusCode()).thenReturn(statusCode);
    when(platform.getClient(anyString())).thenThrow(responseException);

    thrown.expect(ClientsRepositoryException.class);
    thrown.expectMessage(format("Invalid client with id '%s'.", CLIENT_ID));

    repository.getBy(CLIENT_ID);
  }

  @Test
  public void connectionException() throws ClientsRepositoryException {
    int statusCode = 404;
    String message = "message";

    HttpResponseException responseException = mock(HttpConnectionException.class);
    when(responseException.statusCode()).thenReturn(statusCode);
    when(responseException.getMessage()).thenReturn(message);
    when(platform.getClient(anyString())).thenThrow(responseException);

    thrown.expect(ClientsRepositoryException.class);
    thrown
        .expectMessage(format("Invalid client with id '%s'.", CLIENT_ID));

    repository.getBy(CLIENT_ID);
  }

  private Client adapt(CoreServicesClientDto dto) {
    return Client.builder().withId(dto.getId()).withName(dto.getName()).withSecret(dto.getSecret()).build();
  }

}
